#!/usr/bin/env bash

set -o errexit
set -o pipefail

REQ_PATH="$(dirname -- "$0")"/requirements
DASK="-r $REQ_PATH/dask.txt"
JUPYTER="-r $REQ_PATH/jupyter.txt"
COMMON="-r $REQ_PATH/common.txt"

DEPS_DIR=recipe-deps
PYTHON_EXE=python3

CURRENT=false
VENV=false
TAR=false
UNTAR=false

function usage()
{
    echo \
"Usage: install.sh [-h] (--current | --venv PATH) [--tar] [--untar]
                  [--python PATH] [--no-jupyter] [--no-dask]

  -h              Display help
  --current       Install packages in the current environment
                  If a venv is active, packages will be installed there.
                  Otherwise, packages will be installed in the system
                  site-packages directory. It enables usage of 'nsys recipe'
                  without having to source a virtual environment. However, new
                  packages risk colliding with existing ones if different
                  versions are required.
  --venv PATH     Install packages in a virtual environment
                  If it doesn't already exist, it is created. It prevents risk
                  of package collision in the current environment but requires
                  the virtual environment to be activated before running
                  'nsys recipe'.
  --tar           Download wheel packages online and tar
  --untar         Untar the wheel packages and install
  --python PATH   Change the python executable (default is '$PYTHON_EXE')
  --no-jupyter    Do not install requirements for Jupyter Notebook
  --no-dask       Do not install requirements for Dask"
}

function die()
{
    echo "$@" >&2
    exit 1
}

function venv_activate()
{
    if [ -z "$VENV_PATH" ] || [ "$VIRTUAL_ENV" = "$VENV_PATH" ]; then
        return
    else
        "$PYTHON_EXE" -m venv "$VENV_PATH" || die "venv not available."
        source "$VENV_PATH"/bin/activate
        "$PYTHON_EXE" -m pip install --upgrade pip
    fi
}

function tar_deps()
{
    "$PYTHON_EXE" -m pip download $DASK $JUPYTER $COMMON -d "$DEPS_DIR"
    tar -czvf "$DEPS_DIR".tar.gz "$DEPS_DIR"
    rm -rf "$DEPS_DIR"
    echo ""$DEPS_DIR".tar.gz created."
}

function untar_deps()
{
    venv_activate
    tar -xzvf "$DEPS_DIR".tar.gz
    "$PYTHON_EXE" -m pip install "$DEPS_DIR"/* --no-index
}

function install_deps()
{
    venv_activate
    "$PYTHON_EXE" -m pip install $DASK $JUPYTER $COMMON
}

while [ $# -gt 0 ]
do
    case "$1" in
        -h|--help)
            usage
            exit 0
            ;;
        -c|--current)
            CURRENT=true
            ;;
        -n|--venv)
            [ $# -ne 2 ] && die "$1 requires a path to the venv."
            VENV=true
            VENV_PATH="$2"
            shift
            ;;
        -t|--tar)
            TAR=true
            ;;
        -u|--untar)
            UNTAR=true
            ;;
        -d|--no-dask)
            unset DASK
            ;;
        -j|--no-jupyter)
            unset JUPYTER
            ;;
        -p|--python)
            [ $# -ne 2 ] && die "$1 requires the python path."
            PYTHON_EXE="$2"
            shift
            ;;
    esac
    shift
done

hash "$PYTHON_EXE" &>/dev/null || die "$PYTHON_EXE not found."
"$PYTHON_EXE" -m pip --version &>/dev/null || die "pip not found."

if $TAR; then
    tar_deps
    exit 0
fi

if [ "$CURRENT" = "$VENV" ]; then
    echo "Error: either '--current' or '--venv PATH' required" >&2
    usage >&2
    exit 1
fi

if $UNTAR; then
    [ -f "$DEPS_DIR".tar.gz ] || die "$DEPS_DIR.tar.gz does not exist. Please download it using the --tar option."
    untar_deps
else
    install_deps
fi

if [ -t 1 ]; then
    GREEN='\033[0;32m'
    NO_COLOR='\033[0m'
fi

echo -e "$GREEN"

if $VENV; then
    echo "Success: Packages were installed in the venv '$VENV_PATH'."
    echo "It is required to be active before running 'nsys recipe'."
elif [ -n "$VIRTUAL_ENV" ]; then
    echo "Success: Packages were installed in the current venv '$VIRTUAL_ENV'."
    echo "It is required to be active before running 'nsys recipe'."
else
    echo "Success: Packages were installed in the system site-packages directory."
fi

echo -ne "$NO_COLOR"
