import importlib
import numpy as np
import pandas as pd
import subprocess

from pathlib import Path

from nsys_recipe.lib import exceptions
from nsys_recipe.lib import nsys_path

def filter_none(dfs):
    """Remove Nones from the dataframe list.

    If the list only contains Nones or empty dataframes, raise an exception.
    """
    dfs = [df for df in dfs if df is not None and len(df) != 0]
    if not dfs:
        raise exceptions.NoDataError
    return dfs

def get_stats_cls(module, class_name):
    """Get the stats class object from name.

    Parameters
    ----------
    module : str
        Stats module name.
    class_name : str
        Stats class name.
    """
    try:
        stats_cls = getattr(importlib.import_module(module), class_name)
    except ModuleNotFoundError as e:
        raise exceptions.StatsModuleNotFoundError(e)
    return stats_cls

def stats_cls_to_df(db, parsed_args, stats_cls, add_report_col=True):
    """Read stats' SQL query into a dataframe.

    Parameters
    ----------
    db : str
        Database path.
    parsed_args : argparse.Namespace
        Parsed arguments.
    stats_cls : object
        Stats class object.
    add_report_col : bool
        If set to True, add a 'Report' column for the database name to the
        dataframe.
    """
    report, exitval, errmsg = stats_cls.Setup(db, parsed_args)
    if report is None:
        if exitval != stats_cls.EXIT_NODATA:
            raise exceptions.StatsInternalError(errmsg)
        # This is more of a warning than an error.
        print(errmsg)
        return None

    df = pd.read_sql(report.get_query(), report.dbcon)
    if add_report_col:
        df['Report'] = report.dbfile_basename

    return df

def nsysrep_to_sqlite(nsysrep):
    """Export .nsys-rep to .sqlite.

    Find and run the Nsys executable with the 'export' option. If the
    executable is not found or the export fails, print an error message
    without exiting the program.
    """
    # The sqlite file already exists.
    sqlite_file = Path(nsysrep).with_suffix('.sqlite')
    if sqlite_file.is_file():
        return sqlite_file

    nsys_exe = nsys_path.find_installed_file("nsys")

    print("Exporting [{}] to [{}]...".format(nsysrep, sqlite_file), flush=True)
    args = [nsys_exe, "export", "--type", "sqlite", "--output", str(sqlite_file), nsysrep]

    try:
        p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, universal_newlines=True)
    except Exception as e:
        print("SKIPPED: {}: {}".format(nsysrep, e))
        return None

    out, err = p.communicate()
    if p.returncode:
        print("SKIPPED: {}: {}".format(nsysrep, err.strip()))
        return None

    return sqlite_file

def find_report(report_column, orig_column, value):
    """Find the report files that contain 'value' in the 'orig_column'.

    Parameters
    ----------
    report_column : series
        Column with report file info.
    orig_column : series
        Column of interest that contains 'value'.
    value : depends on orig_column
        Value to search for.

    Returns
    -------
    report_file(s) : str
        Returns a comma-separated string.
    """
    index_col = np.where(orig_column == value)
    return ', '.join(report_column.iloc[index])

def stddev(group_df, series_dict, n_col_name='Instances'):
    """Calculate the standard deviation out of aggregated values.

    Parameters
    ----------
    group_df : dataframe
        Subset of data sharing a common grouping key. It contains values before
        the overall aggregation.
    series_dict : dict
        Dictionary of series containing the overall aggregated values.
    n_col_name : str
        Name of the column representing population size.
    """
    instance = series_dict[n_col_name].loc[group_df.name]
    if instance <= 1:
        return group_df['StdDev'].iloc[0]

    var_sum = np.dot(group_df[n_col_name] - 1, group_df['StdDev'] ** 2)
    deviation = group_df['Avg'] - series_dict['Avg'].loc[group_df.name]
    dev_sum = np.dot(group_df[n_col_name], deviation ** 2)
    variance = (var_sum + dev_sum) / (instance - 1)
    return (variance ** 0.5).round(1)
