import pandas as pd
import plotly.graph_objects as go

from ipywidgets import Dropdown, interact, HBox, interactive_output

def display_box(df, x=None, **layout_args):
    if x is None:
        x = df['Report']
    # df has multiple index columns.
    elif isinstance(x, pd.MultiIndex):
        x = ['/'.join(i) for i in x]

    fig = go.Figure()
    fig.add_trace(go.Box(
        x=x,
        q1=df['Q1'],
        median=df['Med'],
        q3=df['Q3'],
        lowerfence=df['Min'],
        upperfence=df['Max'],
        sd=df['StdDev']
    ))

    fig.update_layout(**layout_args)
    fig.show()

def display_quartiles(df, **layout_args):
    fig = go.Figure()

    fig.add_trace(go.Scatter(x=df['Report'], y=df['Q1'], name='Q1'))
    fig.add_trace(go.Scatter(x=df['Report'], y=df['Med'], name='Median'))
    fig.add_trace(go.Scatter(x=df['Report'], y=df['Q3'], name='Q3'))

    fig.update_layout(**layout_args)
    fig.show()

def display_table_per_rank(df):
    if df.empty:
        display(df)
        print("Empty DataFrame")
        return

    report_groups = df.groupby('Report')

    def display_report(name):
        report_df = report_groups.get_group(name)
        report_df = report_df.drop(columns=['Report'])
        display(report_df)

    dropdown = Dropdown(
        options=report_groups.groups.keys(),
        description='Report file:',
        layout={'width': 'max-content'}
    )

    interact(display_report, name=dropdown)

def display_stats_per_operation(df, index_col, **layout_args):
    if df.empty:
        display(df)
        print("Empty DataFrame")
        return

    op_groups = df.groupby(index_col)

    def display_operation(name):
        op_df = op_groups.get_group(name)
        display_box(op_df, **layout_args)
        display_quartiles(op_df, **layout_args)

    operations = list(op_groups.groups.keys())

    # Plot is not being displayed for the default dropdown value. If there is
    # only one element, do not create the dropdown. Otherwise, set it to the
    # second element before resetting to the first one.
    if len(operations) > 1:
        dropdown = Dropdown(
            options=operations,
            description='Operation:',
            layout={'width': 'max-content'},
            value=operations[1]
        )
        interact(display_operation, name=dropdown)
        dropdown.value = operations[0]
    elif len(operations) == 1:
        display_operation(operations[0])

def display_stats_per_range(df, **layout_args):
    if df.empty:
        display(df)
        print("Empty DataFrame")
        return

    range_groups = df.groupby(['Range', 'Style'])

    def display_range(name, style):
        range_df = range_groups.get_group((name, style))
        display_box(range_df, **layout_args)
        display_quartiles(range_df, **layout_args)

    ranges, styles = zip(*range_groups.groups.keys())

    # Plot is not being displayed for the default dropdown value. If there is
    # only one element, do not create the dropdown. Otherwise, set it to the
    # second element before resetting to the first one.
    if len(ranges) > 1:
        range_dropdown = Dropdown(
            options=sorted(set(ranges)),
            description='Range:',
            layout={'width': 'max-content'},
            value=ranges[1]
        )
        style_dropdown = Dropdown(
            options=set(styles),
            description='Style:',
            layout={'width': 'max-content'}
        )
        out = interactive_output(
            display_range,
            {"name": range_dropdown, "style": style_dropdown}
        )

        hbox = HBox([range_dropdown, style_dropdown])
        display(hbox, out)
        range_dropdown.value=ranges[0]
    elif len(operations) == 1:
        display_operation(ranges[0], styles[0])
