#! /bin/bash

# SPDX-FileCopyrightText: Copyright (c) 2023 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# SPDX-License-Identifier: Apache-2.0
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

SCRIPT=`readlink -f $0`
HERE=`dirname $SCRIPT`

CONTAINER_NAME=nvidia-devtools-streamer

if [ "$(docker ps -a | grep $CONTAINER_NAME)" ];
then
    CONTAINER_REMOVAL_NAME=${CONTAINER_NAME}-removal
    # docker removal takes some time and proceed asynchroniously.
    # So rename container not to wait for the removal of the container.
    docker rename ${CONTAINER_NAME} ${CONTAINER_REMOVAL_NAME}
    docker stop ${CONTAINER_REMOVAL_NAME}
fi

CONTAINER_IMAGE="${CONTAINER_IMAGE:=nvidia/devtools-streamer-base:1.0}"

# password for the docker access
PASSWORD="${PASSWORD:=nvidia}"

HOST_LAN_IP=`hostname -I | awk '{print $1}'`
# external ip to map to be aware of
# may want to upgrade this to some sort of check-ip service
HOST_IP="${HOST_IP:=$HOST_LAN_IP}"

# external port to map to outside container
HTTP_PORT="${HTTP_PORT:=8080}"

# UDP port which is used for handling incomming WebRTC connections 
CONNECTION_UDP_PORT="${CONNECTION_UDP_PORT:=8081}"

# TCP port which is used for handling incomming WebRTC connections in case
# of blocked UDP traffic. It can be the same port as the TCP one.
FALLBACK_CONNECTION_TCP_PORT="${FALLBACK_CONNECTION_TCP_PORT:=$CONNECTION_UDP_PORT}"

# Resolution and refresh rate after startup.
SCREEN="${SCREEN:=1920x1080@30}"

if [ -z "$DEVTOOL_DIR" ] || [ -z "$DEVTOOL_CMD" ]; then
    echo "Set DEVTOOL_DIR and DEVTOOL_CMD to map a devtool into the container!"
else
    DEVTOOL_DOCKER_ARGS=(
        "-v" "$DEVTOOL_DIR:/mnt/host/devtool"
        "-e" "DEVTOOL_CMD=$DEVTOOL_CMD"
    )
fi

function create_volume_if_not_exists() {
    local volume_name="${1}"
    if [ -d "$volume_name" ]; then
    volume_info=$(docker volume ls | grep "${volume_name}")
        if [[ -z "${volume_info}" ]]; then
            echo "Volume for openh264 not exists. Creating volume. Name: ${volume_name}"
            docker volume create \
                --name "${volume_name}" \
                -d local
        fi
    fi
}

# folder or docker volume name for caching binaries, required for enabling openh264
# inside the containe
OPENH264_BUILD_CACHE_VOLUME_NAME="${OPENH264_BUILD_CACHE_VOLUME_NAME:=${CONTAINER_NAME}-openh264-volume}"
OPENH264_DOCKER_DIR_NAME=/mnt/host/openh264

# USE_OPENH264_BUILD_CACHE=false disabling caching of openh264 binaries
if [ ! "$USE_OPENH264_BUILD_CACHE" = false ] ; then
    create_volume_if_not_exists "$OPENH264_BUILD_CACHE_VOLUME_NAME"

    OPENH264_DOCKER_ARGS=(
        "-e" "OPENH264_DOCKER_DIR_NAME=$OPENH264_DOCKER_DIR_NAME"
        "-v" "$OPENH264_BUILD_CACHE_VOLUME_NAME:$OPENH264_DOCKER_DIR_NAME"
    )
fi

echo -e "Starting container $CONTAINER"

docker run -d --rm --name $CONTAINER_NAME \
 -e NEKO_NAT1TO1=$HOST_IP \
 -p $HTTP_PORT:8080 \
 -e NEKO_PASSWORD=$PASSWORD \
 -e NEKO_PASSWORD_ADMIN=$PASSWORD \
 -e NEKO_SCREEN=$SCREEN \
 -e NEKO_UDPMUX=$CONNECTION_UDP_PORT \
 -e NEKO_TCPMUX=$FALLBACK_CONNECTION_TCP_PORT \
 -p $CONNECTION_UDP_PORT:$CONNECTION_UDP_PORT/udp \
 -p $FALLBACK_CONNECTION_TCP_PORT:$FALLBACK_CONNECTION_TCP_PORT \
 "${OPENH264_DOCKER_ARGS[@]}" \
 "${DEVTOOL_DOCKER_ARGS[@]}" \
 "$@" \
 $CONTAINER_IMAGE

echo -e "Browse to http://$HOST_LAN_IP:$HTTP_PORT for WebRTC session!"
