import inspect
import nsys_recipe
import sys

from pathlib import Path

def is_recipe_subclass(cls):
    return issubclass(cls, nsys_recipe.Recipe) and cls is not nsys_recipe.Recipe

def print_recipe_list():
    print("\nThe following built-in recipes are available:\n")
    for name, value in inspect.getmembers(nsys_recipe, inspect.isclass):
        if is_recipe_subclass(value):
            print("  {} -- {}".format(value.get_script_name(), value.display_name))

def get_recipe_cls(script_name):
    for name, value in inspect.getmembers(nsys_recipe, inspect.isclass):
        if is_recipe_subclass(value) and value.get_script_name() == script_name:
            return value

    return None

def get_parsed_args(recipe_cls):
    parser = recipe_cls.get_argument_parser()
    parser.add_context_arguments()
    return parser.parse_args(sys.argv[2:])

def run_recipe(recipe_cls):
    parsed_args = get_parsed_args(recipe_cls)
    try:
        with nsys_recipe.Context.create_context(parsed_args.mode) as context:
            with recipe_cls(parsed_args) as recipe:
                recipe.run(context)
                if recipe.output_dir is not None:
                    print("Generated:\n    {}".format(recipe.get_output_dir()))
                else:
                    print("SKIPPED: Nothing to do.")
    except nsys_recipe.ModeModuleNotFoundError as e:
        requirements_file = Path(__file__).parent / 'requirements/dask.txt'
        sys.exit("ERROR: {}\nPlease install packages from '{}'.".format(e, requirements_file))
    except nsys_recipe.StatsModuleNotFoundError as e:
        sys.exit("ERROR: {}\nPlease make sure the Stats report exists.".format(e))
    except nsys_recipe.StatsInternalError as e:
        sys.exit("ERROR: Stats encountered an internal error. {}".format(e))
    except nsys_recipe.ValueError as e:
        sys.exit("ERROR: {}".format(e))
    except nsys_recipe.NoDataError:
        # Early exit due to lack of data.
        sys.exit("SKIPPED: No data available.")

def main():
    if len(sys.argv) < 2:
        sys.exit("ERROR: No recipe specified.")

    if '--help-recipes' in sys.argv:
        print_recipe_list()
        sys.exit(0)

    recipe_cls = get_recipe_cls(sys.argv[1])
    if recipe_cls:
        run_recipe(recipe_cls)
    else:
        sys.exit("ERROR: Unknown recipe.")

if __name__ == "__main__":
    main()
