import os
import pandas as pd

from datetime import datetime

from nsys_recipe.lib import helpers
from nsys_recipe.lib import loader
from nsys_recipe.lib import pace
from nsys_recipe.lib import recipe
from nsys_recipe.lib.args import Option
from nsys_recipe.lib.recipe import Context

# DTSP-14650: Code cleanup
class CudaGpuKernPace(recipe.Recipe):
    display_name = 'CUDA GPU Kernel Pacing'
    description = """
    This recipe investigates the progress and consistency of an iteration based
    application.
"""

    @staticmethod
    def mapper_func(nsysrep, parsed_args):
        sqlite_file = helpers.nsysrep_to_sqlite(nsysrep)
        if sqlite_file is None:
            return None

        paceCfg = parsed_args.pace_config

        results = dict()

        tableNames = {
            loader.k_table_session_start : None, 
            loader.k_table_str : None, 
            loader.k_table_cuda_kernel : [
                paceCfg.nameColumn, 
                paceCfg.startColumn,
                paceCfg.endColumn
            ]}

        file=loader.load_file( sqlite_file, tableNames)    
        #print(file.keys())
        
        ranges_df = file.get(loader.k_table_cuda_kernel)
        if (ranges_df is None) or (ranges_df.shape[0] == 0):
            return None

        str_df = file[loader.k_table_str]
        session_start_df = file[loader.k_table_session_start]

        results = dict() 
        results['filePath'] = os.path.splitext(os.path.basename(nsysrep))[0]

        if not pace.mapper(ranges_df, str_df, session_start_df, paceCfg, results):
            return None
        return results

    def save_data(self, data):
        stats_df = data.get('stats')
        if (stats_df is None) or (stats_df.shape[0] == 0):
            return False

        dstPath = self.add_output_file('stats.parquet', 'Table')
        stats_df.to_parquet(dstPath, index=True)

        pace_map = data.get('pace')
        if (pace_map is None) or (len(pace_map) == 0):
            return False

        pace_group_name, pace_group_map = next(iter(pace_map.items()))
        for pace_table_name, pace_table_df in pace_group_map.items():
            if (pace_table_df is None) or (pace_table_df.shape[0] == 0):
                continue
            dstPath = self.add_output_file('pace_' + pace_table_name + '.parquet')
            pace_table_df.to_parquet(dstPath, index=False)

        files_df = data.get('files')
        if (files_df is None) or (files_df.shape[0] == 0):
            return False
        dstPath = self.add_output_file('files.parquet', 'Table')
        files_df.to_parquet(dstPath, index=False)

        return True

    def reducer_func(self, mapper_results_list):
        mapper_results_list = helpers.filter_none(mapper_results_list)
        if (mapper_results_list is None) or (len(mapper_results_list)==0):
            return False

        mapper_results_list.sort(key=lambda x: x.get('filePath'))

        rank = 0
        stats_list = list()
        for mapper_result in mapper_results_list:
            mapper_result['rank'] = rank
            rank += 1
                    

        results = dict() 

        rank = 0
        file_list = dict()
        for mapper_result in mapper_results_list:
            file_list[mapper_result['rank']] = str(mapper_result['filePath'])
            rank += 1
        results['files'] = pd.DataFrame({"File":pd.Series(file_list)})

        pace.reducer(mapper_results_list, None, None, None, self._parsed_args.pace_config, results)

        return self.save_data(results)

    def save_metadata(self):
        self._analysis_dict.update({
            'EndTime': str(datetime.now()),
            'InputReports': self._parsed_args.dir,
            'Outputs': self._output_files
        })
        self.create_analysis_file()

    def run(self, context):
        super().run(context)

        paceCfg = pace.PaceConfig()
        paceCfg.pace_name_list = [self._parsed_args.name]
        paceCfg.nameColumn='shortName'
        paceCfg.startColumn='start'
        paceCfg.endColumn='end'
        paceCfg.durationColumn='duration'
        self._parsed_args.pace_config = paceCfg

        mapper_res = context.wait(context.map(
            self.mapper_func,
            self._parsed_args.dir,
            parsed_args=self._parsed_args
        ))

        if self.reducer_func(mapper_res):
            self.create_notebook('pace.ipynb', 'nsys_pres.py')
            self.save_metadata()

    @classmethod
    def get_argument_parser(cls):
        parser = super().get_argument_parser()

        parser.add_recipe_argument(Option.OUTPUT)
        parser.add_recipe_argument(Option.FORCE_OVERWRITE)
        parser.add_recipe_argument(
            '--name',
            type=str,
            help="Name of the kernel used as delineator between iterations",
            required=True)
        parser.add_recipe_argument(Option.REPORT_DIR, required=True)

        return parser
