import pandas as pd
import sys

import nsysstats

from nsys_recipe.lib import exceptions
from nsys_recipe.lib import helpers

class AnalysisDetails(nsysstats.Report):
    query = """
SELECT
    duration
FROM
    ANALYSIS_DETAILS
"""

    table_checks = {
        'ANALYSIS_DETAILS':
            '{DBFILE} does not contain analysis details.'
    }

    @staticmethod
    def mapper_func(nsysrep, parsed_args):
        sqlite_file = helpers.nsysrep_to_sqlite(nsysrep)
        if sqlite_file is None:
            return None

        return nsysrep, helpers.stats_cls_to_df(
            sqlite_file, parsed_args, AnalysisDetails, add_report_col=False)

    @staticmethod
    def reducer_func(dfs):
        return pd.concat(dfs)

    @staticmethod
    def _filter_mapper_res(mapper_res):
        filtered_dfs = []
        filtered_nsysreps = []

        for nsysrep, df in mapper_res:
            if df is not None:
                filtered_dfs.append(df)
                filtered_nsysreps.append(nsysrep)

        if not filtered_dfs:
            raise exceptions.NoDataError
        return filtered_dfs, filtered_nsysreps

    @staticmethod
    def get_details(context, parsed_args):
        """Get data from the ANALYSIS_DETAILS table for each input file.

        Returns
        -------
        reducer_res : dataframe
            The concatenation of dataframes returned by the SQL query.
        nsysreps : list of str
            The list of report files with non-empty data. This can be used
            instead of parsed_args.dir to avoid duplicated warning/error
            messages when re-querying ANALYSIS_DETAILS afterwards.
        """
        mapper_res = context.wait(context.map(
            AnalysisDetails.mapper_func, parsed_args.dir, parsed_args=parsed_args))
        dfs, nsysreps = AnalysisDetails._filter_mapper_res(mapper_res)
        reducer_res = AnalysisDetails.reducer_func(dfs)

        return reducer_res, nsysreps
