import argparse
import glob
import os

from enum import Enum

from nsys_recipe.lib import recipe

class Option(Enum):
    """Common recipe options"""
    OUTPUT = 0
    FORCE_OVERWRITE = 1
    REPORT_DIR = 2
    ROWS = 3
    START = 4
    END = 5
    NVTX = 6
    BASE = 7
    MANGLED = 8
    BINS = 9
    DIFF = 10

class ModeAction(argparse.Action):
    def __init__(self, **kwargs):
        kwargs.setdefault('choices',
            tuple(mode.name.replace('_', '-').lower() for mode in recipe.Mode))
        super().__init__(**kwargs)

    def __call__(self, parser, namespace, values, option_string=None):
        value = recipe.Mode[values.replace('-', '_').upper()]
        setattr(namespace, self.dest, value)

class ArgumentParser(argparse.ArgumentParser):
    """Custom argument parser with predefined arguments"""

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self._context_group = self.add_argument_group('Context')
        self._recipe_group = self.add_argument_group('Recipe')

    @property
    def recipe_group(self):
        return self._recipe_group

    @staticmethod
    def _report_directory_path(report_dir):
        files = []

        report_dir = os.path.abspath(report_dir)
        for ext in ('*.nsys-rep', '*.qdrep'):
            files.extend(glob.glob(os.path.join(report_dir, ext)))

        if not files:
            raise argparse.ArgumentTypeError("No nsys-rep files found.")

        return files

    def add_recipe_argument(self, option, *args, **kwargs):
        self.add_argument_to_group(self._recipe_group, option, *args, **kwargs)

    def add_argument_to_group(self, group, option, *args, **kwargs):
        if not isinstance(option, Option):
            group.add_argument(option, *args, **kwargs)
            return

        if option == Option.OUTPUT:
            group.add_argument(
                '--output',
                type=str,
                help="Output directory name",
                **kwargs)
        elif option == Option.FORCE_OVERWRITE:
            group.add_argument(
                '--force-overwrite',
                action='store_true',
                help="Overwrite existing directory",
                **kwargs)
        elif option == Option.REPORT_DIR:
            group.add_argument(
                '--dir',
                type=self._report_directory_path,
                default=None,
                help="Directory of nsys-rep files",
                **kwargs)
        elif option == Option.ROWS:
            group.add_argument(
                '--rows',
                metavar='limit',
                type=int,
                default=-1,
                help="Maximum number of rows per input file",
                **kwargs)
        elif option == Option.START:
            group.add_argument(
                '--start',
                metavar='time',
                type=int,
                help="Start time used for filtering in nanoseconds",
                **kwargs)
        elif option == Option.END:
            group.add_argument(
                '--end',
                metavar='time',
                type=int,
                help="End time used for filtering in nanoseconds",
                **kwargs)
        elif option == Option.NVTX:
            group.add_argument(
                '--nvtx',
                metavar='range[@domain]',
                type=str,
                help="NVTX range and domain used for filtering",
                **kwargs)
        elif option == Option.BASE:
            group.add_argument(
                '--base',
                action='store_true',
                help="Kernel base name",
                **kwargs)
        elif option == Option.MANGLED:
            group.add_argument(
                '--mangled',
                action='store_true',
                help="Kernel mangled name",
                **kwargs)
        elif option == Option.BINS:
            group.add_argument(
                '--bins',
                type=int,
                default=30,
                help="Number of bins",
                **kwargs)
        elif option == Option.DIFF:
            group.add_argument(
                '--diff',
                nargs=2,
                metavar=('DIR1', 'DIR2'),
                help="Two recipe output directories to compare. All other options will be ignored.",
                **kwargs)
        else:
            raise NotImplementedError

    def add_context_arguments(self):
        self._context_group.add_argument(
            '--mode',
            action=ModeAction,
            default=recipe.Mode.CONCURRENT,
            help="Mode to run tasks")
