import os
import platform

from pathlib import Path

from nsys_recipe.lib import exceptions

def get_install_dir():
    """Return the Nsys install directory path.

    If the env var NSYS_DIR is set, return that.
    If not, deduce it from the current file path.
    """
    nsysdir = os.getenv('NSYS_DIR')

    if nsysdir:
        return nsysdir

    parents = Path(__file__).parents
    if len(parents) < 6:
        raise exceptions.ValueError(
            "Please set NSYS_DIR to a valid Nsys install directory that contains internal dependencies.")

    return str(parents[5].resolve())

def get_target_and_host_bin():
    """Return Nsys target and host directory names based on platform."""
    if platform.system() == "Linux":
        if platform.machine() != "x86_64":
            raise NotImplementedError
        host_bin = "host-linux-x64"
        target_bin = "target-linux-x64"
    elif platform.system() == "Windows":
        host_bin = "host-windows-x64"
        target_bin = "target-windows-x64"
    else:
        raise NotImplementedError

    return (target_bin, host_bin)

def find_installed_file(relative_path):
    """Return the full path of the file located in the target or host
    directory."""
    install_dir = get_install_dir()
    candidate_dirs = get_target_and_host_bin() + ('',)
    for candidate_dir in candidate_dirs:
        candidate = os.path.join(install_dir, candidate_dir, relative_path)
        if os.path.exists(candidate):
            return candidate

    raise exceptions.ValueError("Cannot find '{}'."
        " Please set NSYS_DIR to a valid Nsys install directory that contains internal dependencies."
            .format(relative_path))
