const installedApp = {};

const registerExtension = function(i, e, d) {
  installedApp.init = i;
  installedApp.enable = e;
  installedApp.disable = d;
};

function init() {
  if (installedApp.init) {
    installedApp.init();
  }
}
function enable() {
  if (installedApp.enable) {
    installedApp.enable();
  }
}

function disable() {
  if (installedApp.disable) {
    installedApp.disable();
  }
}
 /* Polyfill from Mozilla. */

  // https://tc39.github.io/ecma262/#sec-array.prototype.find
if (!Array.prototype.find && Object.defineProperty) {
  Object.defineProperty(Array.prototype, 'find', {
    value: function(predicate) {
     // 1. Let O be ? ToObject(this value).
      if (this == null) {
        throw new TypeError('"this" is null or not defined');
      }

      var o = Object(this);

      // 2. Let len be ? ToLength(? Get(O, "length")).
      var len = o.length >>> 0;

      // 3. If IsCallable(predicate) is false, throw a TypeError exception.
      if (typeof predicate !== 'function') {
        throw new TypeError('predicate must be a function');
      }

      // 4. If thisArg was supplied, let T be thisArg; else let T be undefined.
      var thisArg = arguments[1];

      // 5. Let k be 0.
      var k = 0;

      // 6. Repeat, while k < len
      while (k < len) {
        // a. Let Pk be ! ToString(k).
        // b. Let kValue be ? Get(O, Pk).
        // c. Let testResult be ToBoolean(? Call(predicate, T, « kValue, k, O »)).
        // d. If testResult is true, return kValue.
        var kValue = o[k];
        if (predicate.call(thisArg, kValue, k, o)) {
          return kValue;
        }
        // e. Increase k by 1.
        k++;
      }

      // 7. Return undefined.
      return undefined;
    }
  });
}

if (!Array.prototype.findIndex && Object.defineProperty) {
  Object.defineProperty(Array.prototype, 'findIndex', {
    value: function(predicate) {
     // 1. Let O be ? ToObject(this value).
      if (this == null) {
        throw new TypeError('"this" is null or not defined');
      }

      var o = Object(this);

      // 2. Let len be ? ToLength(? Get(O, "length")).
      var len = o.length >>> 0;

      // 3. If IsCallable(predicate) is false, throw a TypeError exception.
      if (typeof predicate !== 'function') {
        throw new TypeError('predicate must be a function');
      }

      // 4. If thisArg was supplied, let T be thisArg; else let T be undefined.
      var thisArg = arguments[1];

      // 5. Let k be 0.
      var k = 0;

      // 6. Repeat, while k < len
      while (k < len) {
        // a. Let Pk be ! ToString(k).
        // b. Let kValue be ? Get(O, Pk).
        // c. Let testResult be ToBoolean(? Call(predicate, T, « kValue, k, O »)).
        // d. If testResult is true, return k.
        var kValue = o[k];
        if (predicate.call(thisArg, kValue, k, o)) {
          return k;
        }
        // e. Increase k by 1.
        k++;
      }

      // 7. Return -1.
      return -1;
    }
  });
}
/* Logging
 * Written by Sergey
*/
let debug = false;
/**
 * If called with a false argument, log statements are suppressed.
 */
function setLoggingEnabled(enabled) {
    debug = enabled;
}
/**
 * Log logs the given message using the gnome shell logger (global.log) if the
 * debug variable is set to true.
 *
 * Debug messages may be viewed using the bash command `journalctl
 * /usr/bin/gnome-shell` and grepping the results for 'gTile'.
 */
function log(message) {
    if (debug) {
        global.log("gTile " + message);
    }
}

/* Determine if gnome-shell version newer than required
 * Written by Sergey
*/
function getConfig() {
    return imports.misc.config;
}
const VERSION_34 = { major: 3, minor: 34 };
const VERSION_36 = { major: 3, minor: 36 };
/**
 * ShellVersion is used to parse the version string
 */
class ShellVersion {
    constructor(version) {
        const parts = version.split('.').map((part) => Number(part));
        if (parts.length < 2) {
            throw new Error(`invalid version supplied: ${version}`);
        }
        this.major = parts[0];
        this.minor = parts[1];
        // Tolerate "40.alpha.1" for example. See https://github.com/gTile/gTile/issues/187.
        if (isNaN(this.minor)) {
            this.minor = 0;
        }
        if (isNaN(this.major)) {
            throw new Error(`invalid version supplied: ${JSON.stringify(version)}; got major = ${this.major}, minor = ${this.minor}`);
        }
        this.rawVersion = version;
    }
    static defaultVersion() {
        return ShellVersion.parse(getConfig().PACKAGE_VERSION);
    }
    static parse(version) {
        return new ShellVersion(version);
    }
    version_at_least_34() {
        return versionGreaterThanOrEqualTo(this, VERSION_34);
    }
    version_at_least_36() {
        return versionGreaterThanOrEqualTo(this, VERSION_36);
    }
    print_version() {
        log("Init gnome-shell version " + this.rawVersion + " major " + this.major + " minor " + this.minor);
    }
}
/**
 * Returns true if a is >= b.
 */
function versionGreaterThanOrEqualTo(a, b) {
    return a.major > b.major || (a.major === b.major && a.minor >= b.minor);
}

// Library imports
const Main$1 = imports.ui.main;
const Meta$2 = imports.gi.Meta;
const Shell$1 = imports.gi.Shell;
// Extension imports
const Me$1 = imports.misc.extensionUtils.getCurrentExtension();
const Settings$1 = Me$1.imports.settings;
function bind(keyBindings) {
    log("Binding keys");
    let mySettings = Settings$1.get();
    keyBindings.forEach((callback, key) => {
        //const key = keyString as KeyBindingSettingName;
        if (Main$1.wm.addKeybinding && Shell$1.ActionMode) { // introduced in 3.16
            Main$1.wm.addKeybinding(key, mySettings, Meta$2.KeyBindingFlags.NONE, Shell$1.ActionMode.NORMAL, callback);
        }
        else if (Main$1.wm.addKeybinding && Shell$1.KeyBindingMode) { // introduced in 3.7.5
            Main$1.wm.addKeybinding(key, mySettings, Meta$2.KeyBindingFlags.NONE, Shell$1.KeyBindingMode.NORMAL | Shell$1.KeyBindingMode.MESSAGE_TRAY, callback);
        }
        else {
            global.display.add_keybinding(key, mySettings, Meta$2.KeyBindingFlags.NONE, callback);
        }
    });
}
function unbind(keyBindings) {
    log("Unbinding keys");
    for (let key of keyBindings.keys()) {
        if (Main$1.wm.removeKeybinding) { // introduced in 3.7.2
            Main$1.wm.removeKeybinding(key);
        }
        else {
            global.display.remove_keybinding(key);
        }
    }
}

/* Edited by Sergey after
https://github.com/tpyl/gssnaptoneighbors
 by Timo Pylvanainen <tpyl@iki.fi>
 */
const Meta$1 = imports.gi.Meta;
const WorkspaceManager$1 = global.screen || global.workspace_manager;
const OVERLAP_TOLERANCE = 5;
const SCAN_BOX_SIZE = 50;
/**
 * Return all windows on the currently active workspace
 */
function getWindowsOnActiveWorkspace() {
    let windows = [];
    let windowActors = global.get_window_actors();
    let curWorkSpace = WorkspaceManager$1.get_active_workspace();
    for (let i = 0; i < windowActors.length; i++) {
        let win = windowActors[i].meta_window;
        if (win.located_on_workspace(curWorkSpace) &&
            !win.minimized &&
            win.get_frame_type() == 0) {
            windows.push(win);
        }
    }
    return windows;
}
/**
 * Find the maximum horzontal expansion from x and
 * returns minx, maxx. The initial maximum x is given
 * as argument, and the expansion is never larger than
 * that.
 *
 * The upper and lower limits define the y coordinate
 * range to check for overlapping windows.
 */
function expandHorizontally(x, upper, lower, minx, maxx, windows) {
    for (let i = 0; i < windows.length; i++) {
        let rect = windows[i].get_frame_rect();
        let wt = rect.y;
        let wb = rect.y + rect.height;
        let wl = rect.x;
        let wr = rect.x + rect.width;
        // Check only  if the window overlaps vertically
        if (wb > upper && wt < lower) {
            if (wr < x && minx < wr) {
                minx = wr;
            }
            if (wl > x && wl < maxx) {
                maxx = wl;
            }
        }
    }
    return { min: minx, max: maxx };
}
/**
 * Find the maximum vertical expansion from  y, and
 * returns miny, maxy. The initial maximum y is given
 * as argument, and the expansion is never larger than
 * that.
 *
 * The left and right limits define the x coordinate
 * range to check for overlapping windows.
 */
function expandVertically(y, left, right, miny, maxy, windows) {
    for (let i = 0; i < windows.length; i++) {
        let rect = windows[i].get_frame_rect();
        let wt = rect.y;
        let wb = rect.y + rect.height;
        let wl = rect.x;
        let wr = rect.x + rect.width;
        // Check only  if the window overlaps horizontally
        if (wr > left && wl < right) {
            if (wb < y && miny < wb) {
                miny = wb;
            }
            if (wt > y && wt < maxy) {
                maxy = wt;
            }
        }
    }
    return { min: miny, max: maxy };
}
/**
 * Resize & move the *window* so it touches adjacent windows or
 * screen edge top, bottom, left and right. The top-left corner
 * of the window defines the expansion point.
 *
 * There is an L-ambiguity where the window could be expanded
 * both vertically and horizontally. The expnasion that results
 * in closer to 1 aspect ratio is selected.
 */
function snapToNeighbors(window) {
    log("snapToNeighbors " + window.get_title());
    // Unmaximize first
    if (window.maximized_horizontally || window.maximized_vertically)
        window.unmaximize(Meta$1.MaximizeFlags.HORIZONTAL | Meta$1.MaximizeFlags.VERTICAL);
    let workArea = window.get_work_area_current_monitor();
    let myrect = window.get_frame_rect();
    let windows = getWindowsOnActiveWorkspace();
    // Scan for overlapping windows in a thin bar around the top of the
    // window. The vertical height of the window will be adjusted later.
    let maxHorizw = expandHorizontally(myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2), myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2), myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2) + SCAN_BOX_SIZE, workArea.x, workArea.x + workArea.width, windows);
    let maxHorizh = expandVertically(myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2), maxHorizw.min + OVERLAP_TOLERANCE, maxHorizw.max - OVERLAP_TOLERANCE, workArea.y, workArea.y + workArea.height, windows);
    let maxVerth = expandVertically(myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2), myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2), myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2) + SCAN_BOX_SIZE, workArea.y, workArea.y + workArea.height, windows);
    let maxVertw = expandHorizontally(myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2), maxVerth.min + OVERLAP_TOLERANCE, maxVerth.max - OVERLAP_TOLERANCE, workArea.x, workArea.x + workArea.width, windows);
    if ((maxHorizw.max - maxHorizw.min) * (maxHorizh.max - maxHorizh.min) >
        (maxVertw.max - maxVertw.min) * (maxVerth.max - maxVerth.min)) {
        window.move_resize_frame(true, maxHorizw.min, maxHorizh.min, maxHorizw.max - maxHorizw.min, maxHorizh.max - maxHorizh.min);
    }
    else {
        window.move_resize_frame(true, maxVertw.min, maxVerth.min, maxVertw.max - maxVertw.min, maxVerth.max - maxVerth.min);
    }
}

/**
 * TileSpec represents a rectangular area on display by means of specifying a
 * number of evenly spaced tiles and two corners.
 */
class TileSpec {
    constructor(gridWidth, gridHeight, luc, rdc) {
        this.gridWidth = gridWidth;
        this.gridHeight = gridHeight;
        this.luc = luc;
        this.rdc = rdc;
    }
    toString() {
        return ` ${[this.gridWidth, this.gridHeight].join('x')} ${this.luc.toString()} ${this.rdc.toString()}`;
    }
    toFrameRect(workArea) {
        const elemSize = new Size(workArea.size.width / this.gridWidth, workArea.size.height / this.gridHeight);
        let left;
        let top;
        let right;
        let bottom;
        if (this.luc.types.x == 'tile') {
            const positiveTileNumber = this._convertNegativeCoord(this.gridWidth, this.luc.xy.x);
            left = Math.round(workArea.origin.x + (positiveTileNumber - 1) * elemSize.width);
        }
        else if (this.luc.types.x == 'approx_percentage') {
            const snappedToGrid = Math.round(this.gridWidth * this.luc.xy.x);
            left = Math.round(workArea.origin.x + snappedToGrid * elemSize.width);
        }
        else {
            left = Math.round(workArea.origin.x + workArea.size.width * this.luc.xy.x);
        }
        if (this.luc.types.y == 'tile') {
            const positiveTileNumber = this._convertNegativeCoord(this.gridHeight, this.luc.xy.y);
            top = Math.round(workArea.origin.y + (positiveTileNumber - 1) * elemSize.height);
        }
        else if (this.luc.types.y == 'approx_percentage') {
            const snappedToGrid = Math.round(this.gridHeight * this.luc.xy.y);
            top = Math.round(workArea.origin.y + snappedToGrid * elemSize.height);
        }
        else {
            top = Math.round(workArea.origin.y + workArea.size.height * this.luc.xy.y);
        }
        if (this.rdc.types.x == 'tile') {
            const positiveTileNumber = this._convertNegativeCoord(this.gridWidth, this.rdc.xy.x);
            right = Math.round(workArea.origin.x + positiveTileNumber * elemSize.width);
        }
        else if (this.rdc.types.x == 'approx_percentage') {
            const snappedToGrid = Math.round(this.gridWidth * this.rdc.xy.x);
            right = Math.round(workArea.origin.x + snappedToGrid * elemSize.width);
        }
        else {
            right = Math.round(workArea.origin.x + workArea.size.width * this.rdc.xy.x);
        }
        if (this.rdc.types.y == 'tile') {
            const positiveTileNumber = this._convertNegativeCoord(this.gridHeight, this.rdc.xy.y);
            bottom = Math.round(workArea.origin.y + positiveTileNumber * elemSize.height);
        }
        else if (this.rdc.types.y == 'approx_percentage') {
            const snappedToGrid = Math.round(this.gridHeight * this.rdc.xy.y);
            bottom = Math.round(workArea.origin.y + snappedToGrid * elemSize.height);
        }
        else {
            bottom = Math.round(workArea.origin.y + workArea.size.height * this.rdc.xy.y);
        }
        return new Rect(new XY(left, top), new Size(right - left - 1, bottom - top - 1));
    }
    get gridSize() {
        return new GridSize(this.gridWidth, this.gridHeight);
    }
    isFullscreen(workArea) {
        return this.toFrameRect(workArea).equal(workArea, 1);
    }
    /**
     * Converts negative coordinates (e.g. -1:-1) to a positive format on a specified grid.
     * If x or y is a positive number, it is ignored.
     * E.g. -1:-1 on a 3:3 grid is a 3:3, as well as -1:3.
     */
    _convertNegativeCoord(gridEdges, coord) {
        if (coord < 0) {
            return gridEdges + coord + 1;
        }
        else {
            return coord;
        }
    }
}
const MAX_TUPLE_MEMBER_VALUE = Number.MAX_SAFE_INTEGER;
/**
 * Tuple Holder represents a single starting or ending point (x and y coordinates),
 * as well as the type of the coordinate - "tile", "approx_percentage" or "percentage" now.
 *
 * E.g. ~0.75:0.75 is {X:0.75,Y:0.75}, types - 'percentage' & 'percentage'
 * approximate - true.
 */
class TupleHolder {
    constructor(raw) {
        this.raw = raw;
        const gssk = this.raw.split(':');
        this._validateTuple(gssk);
        this.xy = this._parseTuple(gssk);
        this.types = this._parseTypes(gssk);
    }
    toString() {
        return this.raw;
    }
    _parseTuple(tuple) {
        const x = this._parseCoordinate(tuple[0]);
        const y = this._parseCoordinate(tuple[1]);
        return new XY(x, y);
    }
    _parseTypes(tuple) {
        const typeX = this._parseType(tuple[0]);
        const typeY = this._parseType(tuple[1]);
        return new CoordinateTypesHolder(typeX, typeY);
    }
    _parseCoordinate(coord) {
        return Number(coord.replace('~', ''));
    }
    _parseType(coord) {
        if (coord.includes('~')) {
            return 'approx_percentage';
        }
        else if (coord.includes('.')) {
            return 'percentage';
        }
        else {
            return 'tile';
        }
    }
    _validateTuple(gssk) {
        if (gssk.length !== 2) {
            throw new Error(`Failed to split ${this.raw} into two numbers`);
        }
        this._validateCoordinate(gssk[0]);
        this._validateCoordinate(gssk[1]);
    }
    /**
     * Allowed values:
     * 1.0 (exact match)
     * Any float from 0.0 till 0.999..., with or without preceding approx indicator (~)
     * Any positive or negative integer, except 0
     */
    _validateCoordinate(coord) {
        const testRegex = /(~?0\.[0-9]+|1\.0|-?[1-9]+[0-9]*)/;
        if (!testRegex.test(coord)) {
            throw new Error(`Failed to parse ${coord} in tuple ${this.raw}`);
        }
    }
}
/**
 * Holds coordinate types for the tuple.
 * Currently 3 types are supported - tile, approx_percentage and percentage.
 */
class CoordinateTypesHolder {
    constructor(x, y) {
        this.x = x;
        this.y = y;
    }
}
class GridSize {
    constructor(
    // Number of columns.
    width, 
    // Number of rows.
    height) {
        this.width = width;
        this.height = height;
    }
    toString() {
        return `${this.width}x${this.height}`;
    }
    equals(other) {
        return this.width === other.width && this.height == other.height;
    }
}
function parseGridSizesIgnoringErrors(s) {
    return s.split(',').flatMap((part) => {
        const size = parseGridSizeIgnoringErrors(part.trim());
        return size ? [size] : [];
    });
}
function parseGridSizeIgnoringErrors(s) {
    const parts = s.split("x").map(Number);
    if (parts.length !== 2 || !parts.every(x => !isNaN(x))) {
        return null;
    }
    return new GridSize(parts[0], parts[1]);
}
class XY {
    constructor(x, y) {
        this.x = x;
        this.y = y;
    }
    clone() {
        return new XY(this.x, this.y);
    }
    toString() {
        return 'XY(' + [this.x, this.y].join(', ') + ')';
    }
    dot(b) {
        return this.x * b.x + this.y * b.y;
    }
    unit() {
        const norm = this.l2norm();
        return new XY(this.x / norm, this.y / norm);
    }
    l2norm() {
        return Math.sqrt(this.l2normSquared());
    }
    l2normSquared() {
        return this.dot(this);
    }
    scale(s) {
        return new XY(this.x * s, this.y * s);
    }
    project(b) {
        return b.scale(this.dot(b) / b.l2normSquared());
    }
    scalarProjection(b) {
        return this.dot(b.unit());
    }
    minus(b) {
        return new XY(this.x - b.x, this.y - b.y);
    }
    plus(b) {
        return new XY(this.x + b.x, this.y + b.y);
    }
}
const ADJOIN_DOT_PRODUCT_TOL = .02;
class LineSegment {
    constructor(a, b) {
        this.a = a;
        this.b = b;
    }
    static fromTwoPoints(a, b) {
        return new LineSegment(a, b);
    }
    direction() {
        return this.b.minus(this.a).unit();
    }
    adjoins(other, distTol) {
        return this.parallels(other) && this.lineDistance(other) < distTol;
    }
    parallels(other) {
        const unitDot = this.direction().dot(other.direction());
        return withinTol(Math.abs(unitDot), 1, ADJOIN_DOT_PRODUCT_TOL);
    }
    // The distance between the lines of two line segments. If lines are not
    // (close to) parallel, 0 is returned
    lineDistance(other) {
        return this.perpVectorBetweenLines(other).l2norm();
    }
    // The perpendicular vector between the lines of two line segments. If lines
    // are not (close to) parallel, [0, 0] is returned
    perpVectorBetweenLines(other) {
        const otherDir = other.direction();
        const unitDot = this.direction().dot(otherDir);
        if (!withinTol(Math.abs(unitDot), 1, ADJOIN_DOT_PRODUCT_TOL)) {
            return new XY(0, 0);
        }
        // Basically parallel. Now measure the perpendicular distance between
        // this.a->other.a and other.a->other.b.
        const d = other.a.minus(this.a);
        return d.minus(d.project(otherDir));
    }
}
class Size {
    constructor(width, height) {
        this.width = width;
        this.height = height;
    }
    clone() {
        return new Size(this.width, this.height);
    }
    toString() {
        return [this.width, this.height].join('x');
    }
    area() {
        return this.width * this.height;
    }
}
/**
 * A screen rectangle. A (0, 0) origin represents the top left of a display
 * area. Units are typically pixels.
 */
class Rect {
    constructor(origin, size) {
        this.origin = origin;
        this.size = size;
    }
    clone() {
        return new Rect(this.origin.clone(), this.size.clone());
    }
    toString() {
        return [this.origin, this.size].join(' ');
    }
    equal(r, tol) {
        const close = (a, b) => Math.abs(a - b) <= tol;
        return (close(this.origin.x, r.origin.x) &&
            close(this.origin.y, r.origin.y) &&
            close(this.size.width, r.size.width) &&
            close(this.size.height, r.size.height));
    }
    inset(s) {
        return new Rect(new XY(this.origin.x + s.width, this.origin.y + s.height), new Size(this.size.width - 2 * s.width, this.size.height - 2 * s.height));
    }
    edges() {
        const down = new XY(0, this.size.height);
        const right = new XY(this.size.width, 0);
        const seg = (a, b) => LineSegment.fromTwoPoints(a, b);
        // a---b
        // c---d
        const a = this.origin;
        const b = a.plus(right);
        const c = a.plus(down);
        const d = c.plus(right);
        const rv = new Edges({
            top: seg(a, b),
            right: seg(b, d),
            bottom: seg(c, d),
            left: seg(a, c)
        });
        return rv;
    }
    translate(vec) {
        return new Rect(this.origin.plus(vec), this.size);
    }
    // Increases or decreases the size of the rectangle by moving one of its
    // edges d units along the postive x or y axis, where positive x is right
    // and positive y is down.
    translateEdge(side, d) {
        const [w, h] = [this.size.width, this.size.height];
        switch (side) {
            case Side.Top:
                return new Rect(this.origin.plus(new XY(0, d)), new Size(w, h - d));
            case Side.Bottom:
                return new Rect(this.origin, new Size(w, h + d));
            case Side.Right:
                return new Rect(this.origin, new Size(w + d, h));
            case Side.Left:
                return new Rect(this.origin.plus(new XY(d, 0)), new Size(w - d, h));
            default:
                throw TypeError('bad side type ' + side);
        }
    }
    topLeft() {
        return this.origin;
    }
    topRight() {
        return this.origin.plus(new XY(this.size.width, 0));
    }
    bottomRight() {
        return this.origin.plus(new XY(this.size.width, this.size.height));
    }
    bottomLeft() {
        return this.origin.plus(new XY(0, this.size.height));
    }
    intersection(other) {
        // Not optimized, but that's not necessary.
        const origin = new XY(Math.max(this.topLeft().x, other.topLeft().x), Math.max(this.topLeft().y, other.topLeft().y));
        const br = new XY(Math.min(this.bottomRight().x, other.bottomRight().x), Math.min(this.bottomRight().y, other.bottomRight().y));
        const sizeXY = br.minus(origin);
        const size = new Size(sizeXY.x, sizeXY.y);
        if (size.width < 0 || size.height < 0) {
            return new Rect(new XY(0, 0), new Size(0, 0));
        }
        return new Rect(origin, size);
    }
    valid() {
        return this.size.width >= 0 && this.size.height >= 0;
    }
}
var Side;
(function (Side) {
    Side["Top"] = "TOP";
    Side["Right"] = "RIGHT";
    Side["Bottom"] = "BOTTOM";
    Side["Left"] = "LEFT";
})(Side || (Side = {}));
class Edges {
    constructor(obj) {
        this.top = obj.top;
        this.left = obj.left;
        this.bottom = obj.bottom;
        this.right = obj.right;
    }
    getSide(s) {
        switch (s) {
            case Side.Top: return this.top;
            case Side.Right: return this.right;
            case Side.Bottom: return this.bottom;
            case Side.Left: return this.left;
        }
    }
}
function withinTol(a, b, tol) {
    return Math.abs(a - b) <= tol;
}

/**
 * parsePreset parses a sequence of TileSpec objects from a string like "8x8 0:0
 * 0:7, 8x10 0:0 2:7" or "8x8 0:0 0:7, 0:0 2:7"
 *
 * The 8x8 and 8x10 values above are the "grid size." The grid size may be
 * omitted, then fallback grid size will be used.
 */
function parsePreset(preset, fallback) {
    const parts = preset.split(',').map(x => x.trim());
    let mostRecentSpec = null;
    return parts.map((part, index) => {
        if (hasImpliedGridSize(part)) {
            if (mostRecentSpec === null) {
                if (fallback === undefined) {
                    throw new Error(`preset component[${index}] ${part} of ${preset} is missing grid size (e.g., '3x3') and no fallback is specified`);
                }
                else {
                    part = `${fallback.width}x${fallback.height} ${part}`;
                }
            }
            else {
                part = `${mostRecentSpec.gridWidth}x${mostRecentSpec.gridHeight} ${part}`;
            }
        }
        const parsed = parseSinglePreset(part);
        mostRecentSpec = parsed;
        return parsed;
    });
}
function parseSinglePreset(preset) {
    const ps = preset.trim().split(" ");
    if (ps.length != 3) {
        throw new Error(`Bad preset: ${JSON.stringify(preset)}`);
    }
    const gridFormat = parseTuple(ps[0], "x");
    let luc = new TupleHolder(ps[1]);
    let rdc = new TupleHolder(ps[2]);
    if (gridFormat.x < 1 || gridFormat.x > 100
        || gridFormat.y < 1 || gridFormat.y > 100) {
        throw new Error(`Bad preset: ${JSON.stringify(preset)} grid size out of range 1..100`);
    }
    return new TileSpec(gridFormat.x, gridFormat.y, luc, rdc);
}
function hasImpliedGridSize(singlePreset) {
    return singlePreset.trim().split(" ").length === 2;
}
/**
 * Parses a value like like 6x4 or 1:2 into {X: 6, Y: 4} or {X: 1, Y: 2}.
 */
function parseTuple(unparsed, delim) {
    // parsing grid size in unparsed XdelimY, like 6x4 or 1:2
    const gssk = unparsed.split(delim);
    if (gssk.length !== 2) {
        throw new Error("Failed to split " + unparsed + " by delimiter " + delim + " into two numbers");
    }
    const numbers = gssk.map(Number);
    if (numbers.some(n => isNaN(n) || n > MAX_TUPLE_MEMBER_VALUE)) {
        throw new Error(`All elements of tuple must be intgers in [-inf, ${MAX_TUPLE_MEMBER_VALUE}]: ${JSON.stringify(unparsed)}`);
    }
    return new XY(numbers[0], numbers[1]);
}

/**
 * @fileoverview This file contains incomplete typings for gnome shell types.
 *
 * Probably the best source of definitive API documentation is here:
 * https://gjs-docs.gnome.org/
 *
 * However, there are also some ways the GJS works that make the API docs above
 * slightly incomplete.
 * https://wiki.gnome.org/Projects/GnomeShell/Extensions/StepByStepTutorial
 * mentions that constructors can take a property map as an argument. This file
 * does not correctly type the constructors for these types.
 */
/**
 * From https://gjs-docs.gnome.org/meta4~4_api/meta.frametype.
 */
var FrameType;
(function (FrameType) {
    FrameType[FrameType["NORMAL"] = 0] = "NORMAL";
    FrameType[FrameType["DIALOG"] = 1] = "DIALOG";
    FrameType[FrameType["MODAL_DIALOG"] = 2] = "MODAL_DIALOG";
    FrameType[FrameType["UTILITY"] = 3] = "UTILITY";
    FrameType[FrameType["MENU"] = 4] = "MENU";
    FrameType[FrameType["BORDER"] = 5] = "BORDER";
    FrameType[FrameType["ATTACHED"] = 6] = "ATTACHED";
    FrameType[FrameType["LAST"] = 7] = "LAST";
})(FrameType || (FrameType = {}));
var WindowType;
(function (WindowType) {
    WindowType[WindowType["NORMAL"] = 0] = "NORMAL";
    WindowType[WindowType["DESKTOP"] = 1] = "DESKTOP";
    WindowType[WindowType["DOCK"] = 2] = "DOCK";
    WindowType[WindowType["DIALOG"] = 3] = "DIALOG";
    WindowType[WindowType["MODAL_DIALOG"] = 4] = "MODAL_DIALOG";
    WindowType[WindowType["TOOLBAR"] = 5] = "TOOLBAR";
    WindowType[WindowType["MENU"] = 6] = "MENU";
    WindowType[WindowType["UTILITY"] = 7] = "UTILITY";
    WindowType[WindowType["SPLASHSCREEN"] = 8] = "SPLASHSCREEN";
    WindowType[WindowType["DROPDOWN_MENU"] = 9] = "DROPDOWN_MENU";
    WindowType[WindowType["POPUP_MENU"] = 10] = "POPUP_MENU";
    WindowType[WindowType["TOOLTIP"] = 11] = "TOOLTIP";
    WindowType[WindowType["NOTIFICATION"] = 12] = "NOTIFICATION";
    WindowType[WindowType["COMBO"] = 13] = "COMBO";
    WindowType[WindowType["DND"] = 14] = "DND";
    WindowType[WindowType["OVERRIDE_OTHER"] = 15] = "OVERRIDE_OTHER";
})(WindowType || (WindowType = {}));
var MaximizeFlags;
(function (MaximizeFlags) {
    MaximizeFlags[MaximizeFlags["HORIZONTAL"] = 1] = "HORIZONTAL";
    MaximizeFlags[MaximizeFlags["VERTICAL"] = 2] = "VERTICAL";
    MaximizeFlags[MaximizeFlags["BOTH"] = 3] = "BOTH";
})(MaximizeFlags || (MaximizeFlags = {}));

/*****************************************************************

  This extension has been developed by vibou

  With the help of the gnome-shell community

  Edited by Kvis for gnome 3.8
  Edited by Lundal for gnome 3.18
  Edited by Sergey to add keyboard shortcuts and prefs dialog

 ******************************************************************/
/*****************************************************************
  CONST & VARS
 *****************************************************************/
// Library imports
const St = imports.gi.St;
const Main = imports.ui.main;
const Shell = imports.gi.Shell;
const GObject = imports.gi.GObject;
const PanelMenu = imports.ui.panelMenu;
const Meta = imports.gi.Meta;
const Clutter = imports.gi.Clutter;
const Signals = imports.signals;
imports.ui.workspace;
const Mainloop = imports.mainloop;
// Getter for accesing "get_active_workspace" on GNOME <=2.28 and >= 2.30
const WorkspaceManager = (global.screen || global.workspace_manager);
// Extension imports
const Me = imports.misc.extensionUtils.getCurrentExtension();
const Settings = Me.imports.settings;
// Globals
const SETTINGS_GRID_SIZES = 'grid-sizes';
const SETTINGS_AUTO_CLOSE = 'auto-close';
const SETTINGS_AUTO_CLOSE_KEYBOARD_SHORTCUT = "auto-close-keyboard-shortcut";
const SETTINGS_ANIMATION = 'animation';
const SETTINGS_SHOW_ICON = 'show-icon';
const SETTINGS_GLOBAL_AUTO_TILING = 'global-auto-tiling';
const SETTINGS_GLOBAL_PRESETS = 'global-presets';
const SETTINGS_TARGET_PRESETS_TO_MONITOR_OF_MOUSE = "target-presets-to-monitor-of-mouse";
const SETTINGS_MOVERESIZE_ENABLED = 'moveresize-enabled';
const SETTINGS_WINDOW_MARGIN = 'window-margin';
const SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED = 'window-margin-fullscreen-enabled';
const SETTINGS_MAX_TIMEOUT = 'max-timeout';
const SETTINGS_MAIN_WINDOW_SIZES = 'main-window-sizes';
const SETTINGS_SHOW_GRID_LINES = 'show-grid-lines';
const SETTINGS_INSETS_PRIMARY = 'insets-primary';
const SETTINGS_INSETS_PRIMARY_LEFT = 'insets-primary-left';
const SETTINGS_INSETS_PRIMARY_RIGHT = 'insets-primary-right';
const SETTINGS_INSETS_PRIMARY_TOP = 'insets-primary-top';
const SETTINGS_INSETS_PRIMARY_BOTTOM = 'insets-primary-bottom';
const SETTINGS_INSETS_SECONDARY = 'insets-secondary';
const SETTINGS_INSETS_SECONDARY_LEFT = 'insets-secondary-left';
const SETTINGS_INSETS_SECONDARY_RIGHT = 'insets-secondary-right';
const SETTINGS_INSETS_SECONDARY_TOP = 'insets-secondary-top';
const SETTINGS_INSETS_SECONDARY_BOTTOM = 'insets-secondary-bottom';
const SETTINGS_DEBUG = 'debug';
const SETTINGS_THEME = 'theme';
const gridSettings = {
    [SETTINGS_GRID_SIZES]: [],
    [SETTINGS_AUTO_CLOSE]: null,
    [SETTINGS_AUTO_CLOSE_KEYBOARD_SHORTCUT]: null,
    [SETTINGS_ANIMATION]: null,
    [SETTINGS_SHOW_ICON]: null,
    [SETTINGS_GLOBAL_AUTO_TILING]: null,
    [SETTINGS_GLOBAL_PRESETS]: null,
    [SETTINGS_TARGET_PRESETS_TO_MONITOR_OF_MOUSE]: null,
    [SETTINGS_MOVERESIZE_ENABLED]: null,
    [SETTINGS_WINDOW_MARGIN]: 0,
    [SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED]: false,
    [SETTINGS_MAX_TIMEOUT]: null,
    [SETTINGS_MAIN_WINDOW_SIZES]: [],
    [SETTINGS_SHOW_GRID_LINES]: false,
    [SETTINGS_INSETS_PRIMARY]: 0,
    [SETTINGS_INSETS_PRIMARY_LEFT]: 0,
    [SETTINGS_INSETS_PRIMARY_RIGHT]: 0,
    [SETTINGS_INSETS_PRIMARY_TOP]: 0,
    [SETTINGS_INSETS_PRIMARY_BOTTOM]: 0,
    [SETTINGS_INSETS_SECONDARY]: 0,
    [SETTINGS_INSETS_SECONDARY_LEFT]: 0,
    [SETTINGS_INSETS_SECONDARY_RIGHT]: 0,
    [SETTINGS_INSETS_SECONDARY_TOP]: 0,
    [SETTINGS_INSETS_SECONDARY_BOTTOM]: 0,
    [SETTINGS_DEBUG]: null,
    [SETTINGS_THEME]: null,
};
let launcher;
let tracker;
let nbCols = 0;
let nbRows = 0;
let focusMetaWindow = null;
let focusConnect = false;
let settings = Settings.get();
settings.connect('changed', changed_settings);
let keyControlBound = false;
let enabled = false;
let mainWindowSizes = new Array();
let monitorsChangedConnect = false;
const SHELL_VERSION = ShellVersion.defaultVersion();
const lastResizeInfo = {
    variantIndex: 0,
    lastCallTime: new Date(),
    presetName: '',
    windowTitle: '',
};
let theme = getTheme();
// Hangouts workaround
let excludedApplications = new Array("Unknown");
const keyBindings = new Map([
    ['show-toggle-tiling', () => { globalApp.toggleTiling(); }],
    ['show-toggle-tiling-alt', () => { globalApp.toggleTiling(); }],
]);
const key_bindings_tiling = new Map([
    ['move-left', () => { keyMoveResizeEvent('move', 'left'); }],
    ['move-right', () => { keyMoveResizeEvent('move', 'right'); }],
    ['move-up', () => { keyMoveResizeEvent('move', 'up'); }],
    ['move-down', () => { keyMoveResizeEvent('move', 'down'); }],
    ['resize-left', () => { keyMoveResizeEvent('resize', 'left'); }],
    ['resize-right', () => { keyMoveResizeEvent('resize', 'right'); }],
    ['resize-up', () => { keyMoveResizeEvent('resize', 'up'); }],
    ['resize-down', () => { keyMoveResizeEvent('resize', 'down'); }],
    ['move-left-vi', () => { keyMoveResizeEvent('move', 'left'); }],
    ['move-right-vi', () => { keyMoveResizeEvent('move', 'right'); }],
    ['move-up-vi', () => { keyMoveResizeEvent('move', 'up'); }],
    ['move-down-vi', () => { keyMoveResizeEvent('move', 'down'); }],
    ['resize-left-vi', () => { keyMoveResizeEvent('resize', 'left'); }],
    ['resize-right-vi', () => { keyMoveResizeEvent('resize', 'right'); }],
    ['resize-up-vi', () => { keyMoveResizeEvent('resize', 'up'); }],
    ['resize-down-vi', () => { keyMoveResizeEvent('resize', 'down'); }],
    ['cancel-tiling', () => { keyCancelTiling(); }],
    ['set-tiling', () => { keySetTiling(); }],
    ['change-grid-size', () => { keyChangeTiling(); }],
    ['snap-to-neighbors', () => { snapToNeighborsBind(); }],
    ['snap-to-neighbors', () => { snapToNeighborsBind(); }],
]);
const key_bindings_auto_tiling = new Map([
    ['autotile-main', () => { AutoTileMain('left'); }],
    ['autotile-main-inverted', () => { AutoTileMain('right'); }],
    ['autotile-1', () => { autoTileNCols(1); }],
    ['autotile-2', () => { autoTileNCols(2); }],
    ['autotile-3', () => { autoTileNCols(3); }],
    ['autotile-4', () => { autoTileNCols(4); }],
    ['autotile-5', () => { autoTileNCols(5); }],
    ['autotile-6', () => { autoTileNCols(6); }],
    ['autotile-7', () => { autoTileNCols(7); }],
    ['autotile-8', () => { autoTileNCols(8); }],
    ['autotile-9', () => { autoTileNCols(9); }],
    ['autotile-10', () => { autoTileNCols(10); }],
]);
const key_bindings_presets = new Map([
    ['preset-resize-1', () => { presetResize(1, 'resize1'); }],
    ['preset-resize-2', () => { presetResize(2, 'resize2'); }],
    ['preset-resize-3', () => { presetResize(3, 'resize3'); }],
    ['preset-resize-4', () => { presetResize(4, 'resize4'); }],
    ['preset-resize-5', () => { presetResize(5, 'resize5'); }],
    ['preset-resize-6', () => { presetResize(6, 'resize6'); }],
    ['preset-resize-7', () => { presetResize(7, 'resize7'); }],
    ['preset-resize-8', () => { presetResize(8, 'resize8'); }],
    ['preset-resize-9', () => { presetResize(9, 'resize9'); }],
    ['preset-resize-10', () => { presetResize(10, 'resize10'); }],
    ['preset-resize-11', () => { presetResize(11, 'resize11'); }],
    ['preset-resize-12', () => { presetResize(12, 'resize12'); }],
    ['preset-resize-13', () => { presetResize(13, 'resize13'); }],
    ['preset-resize-14', () => { presetResize(14, 'resize14'); }],
    ['preset-resize-15', () => { presetResize(15, 'resize15'); }],
    ['preset-resize-16', () => { presetResize(16, 'resize16'); }],
    ['preset-resize-17', () => { presetResize(17, 'resize17'); }],
    ['preset-resize-18', () => { presetResize(18, 'resize18'); }],
    ['preset-resize-19', () => { presetResize(19, 'resize19'); }],
    ['preset-resize-20', () => { presetResize(20, 'resize20'); }],
    ['preset-resize-21', () => { presetResize(21, 'resize21'); }],
    ['preset-resize-22', () => { presetResize(22, 'resize22'); }],
    ['preset-resize-23', () => { presetResize(23, 'resize23'); }],
    ['preset-resize-24', () => { presetResize(24, 'resize24'); }],
    ['preset-resize-25', () => { presetResize(25, 'resize25'); }],
    ['preset-resize-26', () => { presetResize(26, 'resize26'); }],
    ['preset-resize-27', () => { presetResize(27, 'resize27'); }],
    ['preset-resize-28', () => { presetResize(28, 'resize28'); }],
    ['preset-resize-29', () => { presetResize(29, 'resize29'); }],
    ['preset-resize-30', () => { presetResize(30, 'resize30'); }],
]);
const keyBindingGlobalResizes = new Map([
    ['action-change-tiling', () => { keyChangeTiling(); }],
    ['action-contract-bottom', () => { keyMoveResizeEvent('contract', 'bottom', true); }],
    ['action-contract-left', () => { keyMoveResizeEvent('contract', 'left', true); }],
    ['action-contract-right', () => { keyMoveResizeEvent('contract', 'right', true); }],
    ['action-contract-top', () => { keyMoveResizeEvent('contract', 'top', true); }],
    ['action-expand-bottom', () => { keyMoveResizeEvent('expand', 'bottom', true); }],
    ['action-expand-left', () => { keyMoveResizeEvent('expand', 'left', true); }],
    ['action-expand-right', () => { keyMoveResizeEvent('expand', 'right', true); }],
    ['action-expand-top', () => { keyMoveResizeEvent('expand', 'top', true); }],
    ['action-move-down', () => { keyMoveResizeEvent('move', 'down', true); }],
    ['action-move-left', () => { keyMoveResizeEvent('move', 'left', true); }],
    ['action-move-right', () => { keyMoveResizeEvent('move', 'right', true); }],
    ['action-move-up', () => { keyMoveResizeEvent('move', 'up', true); }],
    ['action-move-next-monitor', () => { moveWindowToNextMonitor(); }],
    ['action-autotile-main', () => { AutoTileMain('left', true); }],
    ['action-autotile-main-inverted', () => { AutoTileMain('right', true); }],
]);
class App {
    constructor() {
        this.gridsByMonitorKey = {};
        this.gridShowing = false;
        this.gridWidget = null;
        this.gridLinesTimeout = null;
        this.gridTiles = [];
    }
    enable() {
        this.gridShowing = false;
        tracker = Shell.WindowTracker.get_default();
        initSettings();
        const gridWidget = (new St.BoxLayout({ style_class: `${theme}__preview` }));
        this.gridWidget = gridWidget;
        Main.uiGroup.add_actor(gridWidget);
        this.initGrids(gridWidget);
        log("Create Button on Panel");
        launcher = new GTileStatusButton(`${theme}__icon`);
        if (gridSettings[SETTINGS_SHOW_ICON]) {
            Main.panel.addToStatusArea("GTileStatusButton", launcher);
        }
        bind(keyBindings);
        if (gridSettings[SETTINGS_GLOBAL_AUTO_TILING]) {
            bind(key_bindings_auto_tiling);
        }
        if (gridSettings[SETTINGS_GLOBAL_PRESETS]) {
            bind(key_bindings_presets);
        }
        if (gridSettings[SETTINGS_MOVERESIZE_ENABLED]) {
            bind(keyBindingGlobalResizes);
        }
        if (monitorsChangedConnect) {
            Main.layoutManager.disconnect(monitorsChangedConnect);
        }
        log("Connecting monitors-changed");
        monitorsChangedConnect = Main.layoutManager.connect('monitors-changed', () => {
            log("Reinitializing grids on monitors-changed");
            this.destroyGrids();
            this.initGrids(gridWidget);
        });
        enabled = true;
        log("Extention enable completed");
    }
    getGrid(monitor) {
        return this.gridsByMonitorKey[getMonitorKey(monitor)];
    }
    initGrids(gridWidget) {
        log("initGrids");
        log("initGrids nobCols " + nbCols + " nbRows " + nbRows);
        const monitors = activeMonitors();
        for (let monitorIdx = 0; monitorIdx < monitors.length; monitorIdx++) {
            log("New Grid for monitor " + monitorIdx);
            let monitor = monitors[monitorIdx];
            let grid = new Grid(gridWidget, monitorIdx, monitor, "gTile", nbCols, nbRows);
            const key = getMonitorKey(monitor);
            this.gridsByMonitorKey[key] = grid;
            log("initGrids adding grid key " + key);
            // TODO: addChrome is poorly documented. I can't find any reference
            // to it in the gjs-docs site.
            Main.layoutManager.addChrome(grid.actor, { trackFullscreen: true });
            grid.actor.set_opacity(0);
            grid.hide(true);
            log("Connect hide-tiling for monitor " + monitorIdx);
            grid.connectHideTiling = grid.connect('hide-tiling', () => this.hideTiling());
        }
        log("Init grid done");
    }
    destroyGrids() {
        log("destroyGrids");
        for (let gridKey in this.gridsByMonitorKey) {
            const grid = this.gridsByMonitorKey[gridKey];
            grid.hide(true);
            Main.layoutManager.removeChrome(grid.actor);
            log("Disconnect hide-tiling for monitor " + grid.monitor_idx);
            grid.disconnect(grid.connectHideTiling);
            delete this.gridsByMonitorKey[gridKey];
        }
        log("destroyGrids done");
        for (let gridKey in this.gridsByMonitorKey) {
            log("ERROR: gridKey still found in this.gridsByMonitorKey: " + gridKey);
        }
    }
    refreshGrids() {
        log("refreshGrids");
        for (let gridIdx in this.gridsByMonitorKey) {
            const grid = this.gridsByMonitorKey[gridIdx];
            log("refreshGrids calling refresh on " + gridIdx);
            grid.refresh();
        }
        log("refreshGrids done");
    }
    setGridSize(gridSize) {
        nbCols = gridSize.width;
        nbRows = gridSize.height;
        this.refreshGrids();
        if (gridSettings[SETTINGS_SHOW_GRID_LINES]) {
            this._showGridLines(gridSize);
        }
    }
    moveGrids() {
        log("moveGrids");
        if (!this.gridShowing) {
            return;
        }
        let window = focusMetaWindow;
        if (window) {
            for (let gridKey in this.gridsByMonitorKey) {
                let grid = this.gridsByMonitorKey[gridKey];
                let pos_x;
                let pos_y;
                const monitor = grid.monitor;
                if (!monitor) {
                    return;
                }
                if (window.get_monitor() == grid.monitor_idx) {
                    pos_x = window.get_frame_rect().width / 2 + window.get_frame_rect().x;
                    pos_y = window.get_frame_rect().height / 2 + window.get_frame_rect().y;
                    let [mouse_x, mouse_y, mask] = global.get_pointer();
                    let act_x = pos_x - grid.actor.width / 2;
                    let act_y = pos_y - grid.actor.height / 2;
                    if (mouse_x >= act_x
                        && mouse_x <= act_x + grid.actor.width
                        && mouse_y >= act_y
                        && mouse_y <= act_y + grid.actor.height) {
                        log("Mouse x " + mouse_x + " y " + mouse_y +
                            " is inside grid actor rectangle, changing actor X from " + pos_x + " to " + (mouse_x + grid.actor.width / 2) +
                            ", Y from " + pos_y + " to " + (mouse_y + grid.actor.height / 2));
                        pos_x = mouse_x + grid.actor.width / 2;
                        pos_y = mouse_y + grid.actor.height / 2;
                    }
                }
                else {
                    pos_x = monitor.x + monitor.width / 2;
                    pos_y = monitor.y + monitor.height / 2;
                }
                pos_x = Math.floor(pos_x - grid.actor.width / 2);
                pos_y = Math.floor(pos_y - grid.actor.height / 2);
                if (window.get_monitor() == grid.monitor_idx) {
                    pos_x = (pos_x < monitor.x) ? monitor.x : pos_x;
                    pos_x = ((pos_x + grid.actor.width) > (monitor.width + monitor.x)) ? monitor.x + monitor.width - grid.actor.width : pos_x;
                    pos_y = (pos_y < monitor.y) ? monitor.y : pos_y;
                    pos_y = ((pos_y + grid.actor.height) > (monitor.height + monitor.y)) ? monitor.y + monitor.height - grid.actor.height : pos_y;
                }
                let time = (gridSettings[SETTINGS_ANIMATION]) ? 0.3 : 0.1;
                grid.actor.ease({
                    time: time,
                    x: pos_x,
                    y: pos_y,
                    transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                    /*onComplete:updateRegions*/
                });
            }
        }
    }
    updateRegions() {
        var _a;
        /*Main.layoutManager._chrome.updateRegions();*/
        log("updateRegions");
        this.refreshGrids();
        for (let idx in this.gridsByMonitorKey) {
            (_a = this.gridsByMonitorKey[idx].elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
        }
    }
    logState() {
        let count = 0;
        let states = '';
        for (let gridKey in this.gridsByMonitorKey) {
            states += `; ${this.gridsByMonitorKey[gridKey].state()}`;
            count++;
        }
        log(`${count} grids; showing = ${this.gridShowing}: ${states}`);
    }
    showTiling() {
        // TODO(#168): See https://github.com/gTile/gTile/issues/168. Without
        // these two lines, the grid UI does not properly respond to mouseover
        // and other events except for the first time it is shown.
        log("showTiling with fix");
        this.hideTiling();
        this.destroyGrids();
        this.initGrids(this.gridWidget);
        this.logState();
        log("issue#168/showTiling with fix");
        const window = getFocusApp();
        // TODO: remove this global side effect
        focusMetaWindow = window;
        if (!this.gridWidget) {
            return;
        }
        const shouldShowTiling = (() => {
            if (!window) {
                // The tiling UI is for manipulating a particular window.
                return false;
            }
            const wmType = window.get_window_type();
            const layer = window.get_layer();
            return wmType !== WindowType.DESKTOP && layer > 0;
        })();
        if (!shouldShowTiling) {
            return;
        }
        this.gridWidget.visible = true;
        const monitors = activeMonitors();
        for (let monitorIdx = 0; monitorIdx < monitors.length; monitorIdx++) {
            const monitor = monitors[monitorIdx];
            const grid = this.getGrid(monitor);
            if (grid === null) {
                log(`issue#168/showTiling ERROR: did not find grid for monitor ${getMonitorKey(monitor)}`);
                continue;
            }
            let pos_x;
            let pos_y;
            if (window && window.get_monitor() === monitorIdx) {
                // Set pos_x, pos_y to center of the selected window initially.
                pos_x = window.get_frame_rect().width / 2 + window.get_frame_rect().x;
                pos_y = window.get_frame_rect().height / 2 + window.get_frame_rect().y;
                // Old logic with no justifying comments: If the mouse is active
                // and within the rectangle,  set pos_x and pos_y  at mouse
                // position + half the size of the Grid window.
                let [mouse_x, mouse_y, mask] = global.get_pointer();
                let act_x = pos_x - grid.actor.width / 2;
                let act_y = pos_y - grid.actor.height / 2;
                if (mouse_x >= act_x
                    && mouse_x <= act_x + grid.actor.width
                    && mouse_y >= act_y
                    && mouse_y <= act_y + grid.actor.height) {
                    log("Mouse x " + mouse_x + " y " + mouse_y +
                        " is inside grid actor rectangle, changing actor X from " + pos_x + " to " + (mouse_x + grid.actor.width / 2) +
                        ", Y from " + pos_y + " to " + (mouse_y + grid.actor.height / 2));
                    pos_x = mouse_x + grid.actor.width / 2;
                    pos_y = mouse_y + grid.actor.height / 2;
                }
            }
            else {
                pos_x = monitor.x + monitor.width / 2;
                pos_y = monitor.y + monitor.height / 2;
            }
            grid.set_position(Math.floor(pos_x - grid.actor.width / 2), Math.floor(pos_y - grid.actor.height / 2));
            grid.show();
        }
        this.gridShowing = true;
        this.onFocus();
        launcher === null || launcher === void 0 ? void 0 : launcher.activate();
        bindKeyControls();
        this.moveGrids();
    }
    disable() {
        log("Extension disable begin");
        enabled = false;
        // Notice for extension reviewer - this will call Mainloop.RemoveTimeout
        this._hideGridLines();
        if (monitorsChangedConnect) {
            log("Disconnecting monitors-changed");
            Main.layoutManager.disconnect(monitorsChangedConnect);
            monitorsChangedConnect = false;
        }
        unbind(keyBindings);
        unbind(key_bindings_auto_tiling);
        unbind(key_bindings_presets);
        unbind(keyBindingGlobalResizes);
        if (keyControlBound) {
            unbind(key_bindings_tiling);
            keyControlBound = false;
        }
        launcher === null || launcher === void 0 ? void 0 : launcher.destroy();
        launcher = null;
        Main.uiGroup.remove_actor(this.gridWidget);
        this.destroyGrids();
        resetFocusMetaWindow();
        log("Extention disable completed");
    }
    hideTiling() {
        var _a;
        log("hideTiling");
        for (let key in this.gridsByMonitorKey) {
            const grid = this.gridsByMonitorKey[key];
            (_a = grid.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
            grid.hide(false);
        }
        if (this.gridWidget) {
            this.gridWidget.visible = false;
        }
        resetFocusMetaWindow();
        launcher === null || launcher === void 0 ? void 0 : launcher.deactivate();
        this.gridShowing = false;
        unbindKeyControls();
    }
    toggleTiling() {
        if (this.gridShowing) {
            this.hideTiling();
        }
        else {
            this.showTiling();
        }
        return this.gridShowing;
    }
    isGridShowing() {
        return this.gridShowing;
    }
    _hideGridLines(removeTimeout = true) {
        if (this.gridLinesTimeout != null) {
            log("Removing grid lines...");
            if (removeTimeout) {
                Mainloop.timeout_remove(this.gridLinesTimeout);
            }
            this.gridLinesTimeout = null;
            for (let tile of this.gridTiles) {
                Main.uiGroup.remove_actor(tile);
            }
        }
        this.gridTiles = [];
    }
    _showGridLines(gridSize) {
        this._hideGridLines();
        log("Started drawing grid lines...");
        for (let monitorIdx = 0; monitorIdx < activeMonitors().length; monitorIdx++) {
            const workArea = workAreaRectByMonitorIndex(monitorIdx);
            const monitor = activeMonitors()[monitorIdx];
            if (!workArea) {
                continue;
            }
            let tileHeight = workArea.size.height / gridSize.height;
            let tileWidth = workArea.size.width / gridSize.width;
            let topOffset = workArea.topLeft().y;
            let leftOffset = workArea.topLeft().x;
            log(`Starting to draw grid lines for monitor ${JSON.stringify(monitor)}`);
            for (let i = 1; i < gridSize.width; i++) {
                const newGridWidget = new St.BoxLayout({ style_class: `${theme}__grid_lines_preview` });
                const posX = leftOffset + tileWidth * i;
                newGridWidget.width = 1;
                newGridWidget.height = workArea.size.height;
                newGridWidget.x = posX;
                newGridWidget.y = 0;
                this.gridTiles.push(newGridWidget);
                Main.uiGroup.add_actor(newGridWidget);
                log(`Grid vertical line of size ${tileWidth}:${tileHeight} is drawn at ${posX}:0 (monitor offset ${monitor.x}:${monitor.y})`);
            }
            for (let u = 1; u <= gridSize.height; u++) {
                const newGridWidget = new St.BoxLayout({ style_class: `${theme}__grid_lines_preview` });
                const posY = topOffset + tileHeight * u;
                newGridWidget.width = workArea.size.width;
                newGridWidget.height = 1;
                newGridWidget.x = 0;
                newGridWidget.y = posY;
                this.gridTiles.push(newGridWidget);
                Main.uiGroup.add_actor(newGridWidget);
                log(`Grid horizontal line of size ${tileWidth}:${tileHeight} is drawn at 0:${posY} (monitor offset ${monitor.x}:${monitor.y})`);
            }
        }
        this.gridLinesTimeout = Mainloop.timeout_add(1000, () => {
            this._hideGridLines(false);
        });
    }
    /**
     * onFocus is called when the global focus changes.
     */
    onFocus() {
        log("_onFocus ");
        resetFocusMetaWindow();
        const window = getFocusApp();
        if (window && this.gridShowing) {
            log("_onFocus " + window.get_title());
            focusMetaWindow = window;
            let app = tracker.get_window_app(focusMetaWindow);
            let title = focusMetaWindow.get_title();
            const monitors = activeMonitors();
            for (let monitorIdx = 0; monitorIdx < monitors.length; monitorIdx++) {
                let monitor = monitors[monitorIdx];
                let grid = this.getGrid(monitor);
                if (app) {
                    grid === null || grid === void 0 ? void 0 : grid.topbar.setApp(app, title);
                }
                else {
                    grid === null || grid === void 0 ? void 0 : grid.topbar.setTitle(title);
                }
            }
            this.moveGrids();
        }
        else {
            if (this.gridShowing) {
                log("No focus window, hide tiling");
                this.hideTiling();
            }
            else {
                log("tiling window not active");
            }
        }
    }
}
const globalApp = new App();
function changed_settings() {
    log("changed_settings");
    if (enabled) {
        disable();
        enable();
    }
    log("changed_settings complete");
}
const GTileStatusButton = GObject.registerClass({
    GTypeName: 'GTileStatusButton',
}, class GTileStatusButton extends PanelMenu.Button {
    _init(classname) {
        super._init(0.0, "gTile", false);
        //Done by default in PanelMenuButton - Just need to override the method
        if (SHELL_VERSION.version_at_least_34()) {
            const icon = new St.Icon({ style_class: 'system-status-icon' });
            this.add_actor(icon);
            this.add_style_class_name(classname);
            this.connect('button-press-event', this._onButtonPress.bind(this));
        }
        else {
            this.actor.add_style_class_name(classname);
            this.actor.connect('button-press-event', this._onButtonPress.bind(this));
        }
        log("GTileStatusButton _init done");
    }
    reset() {
        this.activated = false;
        if (SHELL_VERSION.version_at_least_34()) {
            this.remove_style_pseudo_class('activate');
        }
        else {
            this.actor.remove_style_pseudo_class('activate');
        }
    }
    activate() {
        if (SHELL_VERSION.version_at_least_34()) {
            this.add_style_pseudo_class('activate');
        }
        else {
            this.actor.add_style_pseudo_class('activate');
        }
    }
    deactivate() {
        if (SHELL_VERSION.version_at_least_34()) {
            this.remove_style_pseudo_class('activate');
        }
        else {
            this.actor.remove_style_pseudo_class('activate');
        }
    }
    _onButtonPress(actor, event) {
        log(`_onButtonPress Click Toggle Status on system panel ${this}`);
        globalApp.toggleTiling();
    }
    _destroy() {
        this.activated = null;
    }
});
/*****************************************************************
  SETTINGS
 *****************************************************************/
function initGridSizes(configValue) {
    let gridSizes = parseGridSizesIgnoringErrors(configValue);
    if (gridSizes.length === 0) {
        gridSizes = [
            new GridSize(8, 6),
            new GridSize(6, 4),
            new GridSize(4, 4),
        ];
    }
    gridSettings[SETTINGS_GRID_SIZES] = gridSizes;
}
function getBoolSetting(settingName) {
    const value = settings.get_boolean(settingName);
    if (value === undefined) {
        log("Undefined settings " + settingName);
        gridSettings[settingName] = false;
        return false;
    }
    else {
        gridSettings[settingName] = value;
    }
    return value;
}
function getIntSetting(settingsValue) {
    let iss = settings.get_int(settingsValue);
    if (iss === undefined) {
        log("Undefined settings " + settingsValue);
        return 0;
    }
    else {
        return iss;
    }
}
function initSettings() {
    log("Init settings");
    theme = getTheme();
    const gridSizes = settings.get_string(SETTINGS_GRID_SIZES) || '';
    log(SETTINGS_GRID_SIZES + " set to " + gridSizes);
    initGridSizes(gridSizes);
    getBoolSetting(SETTINGS_AUTO_CLOSE);
    getBoolSetting(SETTINGS_AUTO_CLOSE_KEYBOARD_SHORTCUT);
    getBoolSetting(SETTINGS_ANIMATION);
    getBoolSetting(SETTINGS_SHOW_ICON);
    getBoolSetting(SETTINGS_GLOBAL_AUTO_TILING);
    getBoolSetting(SETTINGS_GLOBAL_PRESETS);
    getBoolSetting(SETTINGS_TARGET_PRESETS_TO_MONITOR_OF_MOUSE);
    getBoolSetting(SETTINGS_MOVERESIZE_ENABLED);
    getBoolSetting(SETTINGS_SHOW_GRID_LINES);
    gridSettings[SETTINGS_WINDOW_MARGIN] = getIntSetting(SETTINGS_WINDOW_MARGIN);
    gridSettings[SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED] = getBoolSetting(SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED);
    gridSettings[SETTINGS_MAX_TIMEOUT] = getIntSetting(SETTINGS_MAX_TIMEOUT);
    // initialize these from settings, the first set of sizes
    if (nbCols == 0 || nbRows == 0) {
        nbCols = gridSettings[SETTINGS_GRID_SIZES][0].width;
        nbRows = gridSettings[SETTINGS_GRID_SIZES][0].height;
    }
    const mainWindowSizesString = settings.get_string(SETTINGS_MAIN_WINDOW_SIZES);
    log(SETTINGS_MAIN_WINDOW_SIZES + " settings found " + mainWindowSizesString);
    mainWindowSizes = [];
    let mainWindowSizesArray = mainWindowSizesString.split(",");
    for (var i in mainWindowSizesArray) {
        let size = mainWindowSizesArray[i];
        if (size.includes("/")) {
            let fraction = size.split("/");
            let ratio = parseFloat(fraction[0]) / parseFloat(fraction[1]);
            mainWindowSizes.push(ratio);
        }
        else {
            mainWindowSizes.push(parseFloat(size));
        }
    }
    log(SETTINGS_MAIN_WINDOW_SIZES + " set to " + mainWindowSizes);
    log("Init complete, nbCols " + nbCols + " nbRows " + nbRows);
}
function getMonitorTier(monitor) {
    return isPrimaryMonitor(monitor) ? 'primary' : 'secondary';
}
function getMonitorInsets(tier) {
    switch (tier) {
        case 'primary':
            return {
                top: getIntSetting(SETTINGS_INSETS_PRIMARY_TOP),
                bottom: getIntSetting(SETTINGS_INSETS_PRIMARY_BOTTOM),
                left: getIntSetting(SETTINGS_INSETS_PRIMARY_LEFT),
                right: getIntSetting(SETTINGS_INSETS_PRIMARY_RIGHT)
            }; // Insets on primary monitor
        case 'secondary':
            return {
                top: getIntSetting(SETTINGS_INSETS_SECONDARY_TOP),
                bottom: getIntSetting(SETTINGS_INSETS_SECONDARY_BOTTOM),
                left: getIntSetting(SETTINGS_INSETS_SECONDARY_LEFT),
                right: getIntSetting(SETTINGS_INSETS_SECONDARY_RIGHT)
            };
        default:
            throw new Error(`unknown monitor name ${JSON.stringify(tier)}`);
    }
}
function enable() {
    setLoggingEnabled(getBoolSetting(SETTINGS_DEBUG));
    log("Extension enable begin");
    SHELL_VERSION.print_version();
    globalApp.enable();
}
function disable() {
    globalApp.disable();
}
function resetFocusMetaWindow() {
    log("resetFocusMetaWindow");
    focusMetaWindow = null;
}
function reset_window(metaWindow) {
    metaWindow.unmaximize(Meta.MaximizeFlags.HORIZONTAL);
    metaWindow.unmaximize(Meta.MaximizeFlags.VERTICAL);
    metaWindow.unmaximize(Meta.MaximizeFlags.BOTH);
}
function _getInvisibleBorderPadding(metaWindow) {
    let outerRect = metaWindow.get_frame_rect();
    let inputRect = metaWindow.get_buffer_rect();
    let borderX = outerRect.x - inputRect.x;
    let borderY = outerRect.y - inputRect.y;
    return [borderX, borderY];
}
function _getVisibleBorderPadding(metaWindow) {
    let clientRect = metaWindow.get_frame_rect();
    let outerRect = metaWindow.get_frame_rect();
    let borderX = outerRect.width - clientRect.width;
    let borderY = outerRect.height - clientRect.height;
    return [borderX, borderY];
}
function move_maximize_window(metaWindow, x, y) {
    const [borderX, borderY] = _getInvisibleBorderPadding(metaWindow);
    x = x - borderX;
    y = y - borderY;
    metaWindow.move_frame(true, x, y);
    metaWindow.maximize(Meta.MaximizeFlags.HORIZONTAL | Meta.MaximizeFlags.VERTICAL);
}
/**
 * Resizes window considering margin settings
 * @param metaWindow
 * @param x
 * @param y
 * @param width
 * @param height
 */
function moveResizeWindowWithMargins(metaWindow, x, y, width, height) {
    let [borderX, borderY] = _getInvisibleBorderPadding(metaWindow);
    let [vBorderX, vBorderY] = _getVisibleBorderPadding(metaWindow);
    log("move_resize_window_with_margins " + metaWindow.get_title() + " " + x + ":" + y + " - " + width
        + ":" + height + " margin " + gridSettings[SETTINGS_WINDOW_MARGIN] + " borders invisible " +
        borderX + ":" + borderY + " visible " + vBorderX + ":" + vBorderY);
    x = x + gridSettings[SETTINGS_WINDOW_MARGIN];
    y = y + gridSettings[SETTINGS_WINDOW_MARGIN];
    width = width - gridSettings[SETTINGS_WINDOW_MARGIN] * 2;
    height = height - gridSettings[SETTINGS_WINDOW_MARGIN] * 2;
    x = x + vBorderX;
    y = y + vBorderY;
    width = width - 2 * vBorderX;
    height = height - 2 * vBorderY;
    log("After margins and visible border window is " + x + ":" + y + " - " + width + ":" + height);
    metaWindow.move_frame(true, x, y);
    metaWindow.move_resize_frame(true, x, y, width, height);
}
function getNotFocusedWindowsOfMonitor(monitor) {
    const monitors = activeMonitors();
    let windows = global.get_window_actors().filter(function (w) {
        let app = tracker.get_window_app(w.meta_window);
        if (app == null) {
            return false;
        }
        let appName = app.get_name();
        return !contains(excludedApplications, appName)
            && w.meta_window.get_window_type() == Meta.WindowType.NORMAL
            && w.meta_window.get_workspace() == WorkspaceManager.get_active_workspace()
            && w.meta_window.showing_on_its_workspace()
            && monitors[w.meta_window.get_monitor()] == monitor
            && focusMetaWindow != w.meta_window;
    });
    return windows;
}
function getMonitorKey(monitor) {
    return monitor.x + ":" + monitor.width + ":" + monitor.y + ":" + monitor.height;
}
function contains(a, obj) {
    var i = a.length;
    while (i--) {
        if (a[i] === obj) {
            return true;
        }
    }
    return false;
}
/**
 * Get focused window by iterating though the windows on the active workspace.
 * @returns {Object} The focussed window object. False if no focussed window was found.
 */
function getFocusApp() {
    if (tracker.focus_app == null) {
        return null;
    }
    let focusedAppName = tracker.focus_app.get_name();
    if (contains(excludedApplications, focusedAppName)) {
        return null;
    }
    return WorkspaceManager.get_active_workspace().list_windows().find((window) => window.has_focus()) || null;
}
function getFocusWindow() {
    const focus_app = tracker.focus_app;
    if (!focus_app || excludedApplications[focus_app.get_name()]) {
        return null;
    }
    return WorkspaceManager.get_active_workspace().list_windows()
        .find(w => w.has_focus());
}
function activeMonitors() {
    return Main.layoutManager.monitors;
}
/**
 * Determine if the given monitor is the primary monitor.
 * @param {Object} monitor The given monitor to evaluate.
 * @returns {boolean} True if the given monitor is the primary monitor.
 * */
function isPrimaryMonitor(monitor) {
    return Main.layoutManager.primaryMonitor.x == monitor.x && Main.layoutManager.primaryMonitor.y == monitor.y;
}
function getWorkAreaByMonitor(monitor) {
    const monitors = activeMonitors();
    for (let monitor_idx = 0; monitor_idx < monitors.length; monitor_idx++) {
        let mon = monitors[monitor_idx];
        if (mon.x == monitor.x && mon.y == monitor.y) {
            return getWorkArea(monitor, monitor_idx);
        }
    }
    return null;
}
/**
 * @deprecated Use {@link workAreaRectByMonitorIndex} instead.
 */
function getWorkAreaByMonitorIdx(monitor_idx) {
    const monitors = activeMonitors();
    let monitor = monitors[monitor_idx];
    return getWorkArea(monitor, monitor_idx);
}
function workAreaRectByMonitorIndex(monitorIndex) {
    const monitor = activeMonitors()[monitorIndex];
    if (!monitor) {
        return null;
    }
    const waLegacy = getWorkArea(monitor, monitorIndex);
    return (new Rect(new XY(waLegacy.x, waLegacy.y), new Size(waLegacy.width, waLegacy.height)));
}
/**
 * @deprecated Use {@link workAreaRectByMonitorIndex} instead.
 */
function getWorkArea(monitor, monitor_idx) {
    const wkspace = WorkspaceManager.get_active_workspace();
    const work_area = wkspace.get_work_area_for_monitor(monitor_idx);
    const insets = getMonitorInsets(getMonitorTier(monitor));
    return {
        x: work_area.x + insets.left,
        y: work_area.y + insets.top,
        width: work_area.width - insets.left - insets.right,
        height: work_area.height - insets.top - insets.bottom
    };
}
function bindKeyControls() {
    if (!keyControlBound) {
        bind(key_bindings_tiling);
        if (focusConnect) {
            global.display.disconnect(focusConnect);
        }
        focusConnect = global.display.connect('notify::focus-window', () => globalApp.onFocus());
        if (!gridSettings[SETTINGS_GLOBAL_AUTO_TILING]) {
            bind(key_bindings_auto_tiling);
        }
        if (!gridSettings[SETTINGS_GLOBAL_PRESETS]) {
            bind(key_bindings_presets);
        }
        keyControlBound = true;
    }
}
function unbindKeyControls() {
    if (keyControlBound) {
        unbind(key_bindings_tiling);
        if (focusConnect) {
            log("Disconnect notify:focus-window");
            global.display.disconnect(focusConnect);
            focusConnect = false;
        }
        if (!gridSettings[SETTINGS_GLOBAL_AUTO_TILING]) {
            unbind(key_bindings_auto_tiling);
        }
        if (!gridSettings[SETTINGS_GLOBAL_PRESETS]) {
            unbind(key_bindings_presets);
        }
        if (!gridSettings[SETTINGS_MOVERESIZE_ENABLED]) {
            unbind(keyBindingGlobalResizes);
        }
        keyControlBound = false;
    }
}
function keyCancelTiling() {
    log("Cancel key event");
    globalApp.hideTiling();
}
function keySetTiling() {
    var _a, _b;
    log("keySetTiling");
    if (focusMetaWindow) {
        const monitors = activeMonitors();
        let mind = focusMetaWindow.get_monitor();
        let monitor = monitors[mind];
        getMonitorKey(monitor);
        const grid = globalApp.getGrid(monitor);
        log("In grid " + grid);
        (_b = (_a = grid === null || grid === void 0 ? void 0 : grid.elementsDelegate) === null || _a === void 0 ? void 0 : _a.currentElement) === null || _b === void 0 ? void 0 : _b._onButtonPress();
    }
}
function keyChangeTiling() {
    log("keyChangeTiling. Current nbCols " + nbCols + " nbRos " + nbRows);
    const gridSizes = gridSettings[SETTINGS_GRID_SIZES];
    if (gridSizes.length === 0) {
        return;
    }
    const currentIndex = gridSizes.findIndex((size) => size.width === nbCols && size.height === nbRows);
    const newIndex = currentIndex === -1 ? 0 : (currentIndex + 1) % gridSizes.length;
    const newGridSize = gridSizes[newIndex];
    globalApp.setGridSize(newGridSize);
    setInitialSelection();
}
function setInitialSelection() {
    if (!focusMetaWindow) {
        return;
    }
    let mind = focusMetaWindow.get_monitor();
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    const grid = globalApp.getGrid(monitor);
    if (!grid) {
        log("no grid widget available in setInitialSelection");
        return;
    }
    grid.setInitialSelection(focusMetaWindow);
}
function keyMoveResizeEvent(type, key, is_global = false) {
    if (is_global) {
        focusMetaWindow = getFocusApp();
    }
    log("Got key event " + type + " " + key);
    if (!focusMetaWindow) {
        return;
    }
    log("Going on..");
    let mind = focusMetaWindow.get_monitor();
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    const grid = globalApp.getGrid(monitor);
    if (!grid) {
        return;
    }
    if (!grid.moveResize(focusMetaWindow, type, key)) {
        return;
    }
    if (is_global) {
        keySetTiling();
    }
}
/**
 * Resize window to the given preset.
 * @param  {number}  Identifier of the resize preset (1 - 30)
 */
function presetResize(presetName, settingName) {
    // Check if there's a focusable window
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut");
        return;
    }
    // Lets assume window titles are always unique.
    // Note: window roles 'window.get_role()' would offer a unique identifier.
    // Unfortunately role is often set to NULL.
    log("presetResize window title: " + window.get_title());
    // Ensure that the window is not maximized
    reset_window(window);
    // Fetch, parse and validate the given preset.
    // Expected preset format is "XxY x1:y1 x2:y2[,x1:y1 x2:y2]":
    //  - XxY is grid size like 6x8
    //  - x1:y1 is left upper corner tile coordinates in grid tiles, starting from 0
    //  - x2:y2 is right down corner tile coordinates in grid tiles
    //  - a preset can define multiple variants (e.g. "3x2 0:0 0:1,0:0 1:1,0:0 2:1")
    //  - variants can be activated using the same shortcut consecutively
    const presetString = settings.get_string(settingName) || '';
    log("Preset resize " + presetName + "  is " + presetString);
    let tileSpecs = [];
    try {
        tileSpecs = parsePreset(presetString, currentGridSize());
    }
    catch (err) {
        log(`Bad preset ${presetName} ${JSON.stringify(presetString)}: ${err}`);
        return;
    }
    if (tileSpecs.length === 0) {
        return;
    }
    // parse the preset string (grid size, left-upper-corner, right-down-corner)
    tileSpecs[0].gridSize;
    // handle preset variants (if there are any)
    let variantCount = tileSpecs.length;
    if (variantCount > 1) {
        if ((lastResizeInfo.lastCallTime.getTime() + gridSettings[SETTINGS_MAX_TIMEOUT]) > new Date().getTime() &&
            lastResizeInfo.presetName === presetName.toString() &&
            lastResizeInfo.windowTitle == window.get_title()) {
            // within timeout (default: 2s), same preset & same window:
            // increase variant counter, but consider upper boundary
            lastResizeInfo.variantIndex = (lastResizeInfo.variantIndex + 1) % variantCount;
        }
        else {
            // timeout, new preset or new window:
            // update presetState.last_preset and reset variant counter
            lastResizeInfo.variantIndex = 0;
        }
    }
    else {
        lastResizeInfo.variantIndex = 0;
    }
    // retrieve current preset variant
    const tileSpec = tileSpecs[lastResizeInfo.variantIndex];
    // target monitor of current window
    let monitorIdx = window.get_monitor();
    // optionally target monitor where curser is currently
    if (gridSettings[SETTINGS_TARGET_PRESETS_TO_MONITOR_OF_MOUSE]) {
        const [mouseX, mouseY, mask] = global.get_pointer();
        log(`current mouse position ${mouseX}, ${mouseY}`);
        const monitors = activeMonitors();
        log(`monitors: ${JSON.stringify(monitors)}`);
        for (let idx = 0; idx < monitors.length; idx++) {
            let monitor = monitors[idx];
            if (mouseX >= monitor.x && mouseX <= monitor.x + (monitor.x + monitor.width)
                && mouseY >= monitor.y && mouseY <= (monitor.y + monitor.height)) {
                monitorIdx = idx;
            }
        }
    }
    // do the maths to resize the window
    const workArea = workAreaRectByMonitorIndex(monitorIdx);
    if (!workArea) {
        log(`Failed to get active work area for window while performing preset ${presetName} ${JSON.stringify(presetString)}`);
        return;
    }
    // The rectangle already incorporates insets, but it doesn't incorporate
    // window margin.
    const zeroMargins = tileSpec.isFullscreen(workArea) && !getBoolSetting(SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED);
    const marginSize = new Size(zeroMargins ? 0 : gridSettings[SETTINGS_WINDOW_MARGIN], zeroMargins ? 0 : gridSettings[SETTINGS_WINDOW_MARGIN]);
    const rect = tileSpec.toFrameRect(workArea).inset(marginSize);
    moveWindowToRect(window, rect);
    lastResizeInfo.presetName = presetName.toString();
    lastResizeInfo.windowTitle = window.get_title();
    lastResizeInfo.lastCallTime = new Date();
    if (gridSettings[SETTINGS_AUTO_CLOSE_KEYBOARD_SHORTCUT]) {
        globalApp.hideTiling();
    }
}
// Move the window to the next monitor.
function moveWindowToNextMonitor() {
    log("moveWindowToNextMonitor");
    let window = getFocusWindow();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut to move window");
        return;
    }
    reset_window(window);
    const numMonitors = activeMonitors().length;
    if (numMonitors == 0) {
        return;
    }
    const ts = parsePreset("5x5 1:1 3:3")[0];
    const srcMonitorIndex = window.get_monitor();
    const dstMonitorIndex = (srcMonitorIndex + 1) % numMonitors;
    const workArea = workAreaRectByMonitorIndex(dstMonitorIndex);
    if (!workArea) {
        log(`Failed to get active work area for window while moving it to the next monitor.`);
        return;
    }
    const rect = ts.toFrameRect(workArea);
    moveWindowToRect(window, rect);
}
/**
 * Moves a window to the specified region. This may resize the window as well as
 * move its origin.
 */
function moveWindowToRect(window, rect) {
    // The move_frame line is a workaround for a bug affecting gnome terminal
    // windows. See https://github.com/gTile/gTile/issues/176#issue-751198339.
    window.move_frame(true, rect.origin.x, rect.origin.y);
    window.move_resize_frame(true, rect.origin.x, rect.origin.y, rect.size.width, rect.size.height);
}
// Converts global nbCols & nbRows to GridSize object
function currentGridSize() {
    return new GridSize(nbCols, nbRows);
}
/*****************************************************************
  PROTOTYPES
 *****************************************************************/
class TopBar {
    constructor(_title) {
        this._title = _title;
        this.actor = new St.BoxLayout({
            style_class: `${theme}__title-container`
        });
        this._stlabel = new St.Label({
            style_class: `${theme}__title`,
            text: this._title
        });
        this._closebutton = new St.Button({
            style_class: `${theme}__close`,
        });
        this._closebutton.add_style_class_name(`${theme}__close-container`);
        this._connect_id = this._closebutton.connect('button-press-event', this._onButtonPress);
        this.actor.add_child(this._closebutton);
        this.actor.add_child(this._stlabel);
    }
    setTitle(title) {
        this._title = title;
        this._stlabel.text = this._title;
    }
    setApp(app, title) {
        this._title = app.get_name() + " - " + title;
        log("title: " + this._title);
        this._stlabel.text = this._title;
    }
    _onButtonPress() {
        globalApp.hideTiling();
    }
    destroy() {
        this._closebutton.disconnect(this._connect_id);
    }
}
class ToggleSettingsButtonListener {
    constructor() {
        this.actors = [];
    }
    addActor(actor) {
        log("ToggleSettingsButtonListener Connect update-toggle");
        actor.connect('update-toggle', this._updateToggle.bind(this));
        this.actors.push(actor);
    }
    _updateToggle() {
        log("ToggleSettingsButtonListener _updateToggle");
        for (let actor of this.actors) {
            actor._update();
        }
    }
}
function ToggleSettingsButton(text, property) {
    this._init(text, property);
}
ToggleSettingsButton.prototype = {
    _init: function (text, property) {
        this.text = text;
        this.actor = new St.Button({
            style_class: `${theme}__action-button`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.label = new St.Label({
            style_class: `${theme}__action-label`,
            reactive: true,
            can_focus: true,
            track_hover: true,
            text: this.text
        });
        this.icon = new St.BoxLayout({
            style_class: `${theme}__action-button--${this.text}`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.actor.add_actor(this.icon);
        this.property = property;
        this._update();
        log("ToggleSettingsButton Connect button-press-event");
        this.actor.connect('button-press-event', this._onButtonPress.bind(this));
        log("ToggleSettingsButton Connect update-toggle");
        this.connect('update-toggle', this._update.bind(this));
    },
    _update: function () {
        log("ToggleSettingsButton _update event " + this.property);
        if (gridSettings[this.property]) {
            this.actor.add_style_pseudo_class('activate');
        }
        else {
            this.actor.remove_style_pseudo_class('activate');
        }
    },
    _onButtonPress: function () {
        gridSettings[this.property] = !gridSettings[this.property];
        log("ToggleSettingsButton _onButtonPress " + this.property + ": " + gridSettings[this.property] + ", emitting signal update-toggle");
        this.emit('update-toggle');
    }
};
Signals.addSignalMethods(ToggleSettingsButton.prototype);
const ACTION_CLASSES = {
    BUTTON: "__action-button",
    MAIN_AND_LIST: "__action-button--main-and-list",
    TWO_LIST: "__action-button--two-list",
};
class ActionButton {
    constructor(grid, classname) {
        this.grid = grid;
        this.grid = grid;
        this.actor = new St.Button({
            style_class: `${getTheme()}${ACTION_CLASSES.BUTTON}`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.icon = new St.BoxLayout({ style_class: classname, reactive: true, can_focus: true, track_hover: true });
        this.actor.add_actor(this.icon);
        log("ActionButton Connect button-press-event");
        this.actor.connect('button-press-event', this._onButtonPress.bind(this));
    }
    _onButtonPress() {
        log("ActionButton _onButtonPress Emitting signal button-press-event");
        this.emit('button-press-event');
    }
    /** Functions replaced by Signals.addSignalMethods. */
    connect(eventName, handler) { return 0; }
    disconnect(id) { }
    emit(name, ...args) { }
}
Signals.addSignalMethods(ActionButton.prototype);
class AutoTileMainAndList extends ActionButton {
    constructor(grid) {
        const theme = `${getTheme()}${ACTION_CLASSES.MAIN_AND_LIST}`;
        super(grid, theme);
        this.classname = theme;
        log("AutoTileMainAndList connect button-press-event");
        this.connect('button-press-event', () => this._onButtonPress());
    }
    _onButtonPress() {
        AutoTileMain();
        log("AutoTileMainAndList _onButtonPress Emitting signal resize-done");
        this.emit('resize-done');
    }
}
Signals.addSignalMethods(AutoTileMainAndList.prototype);
function AutoTileMain(tile_direction = 'left', is_global = false) {
    let preset = "AutoTileMain-" + tile_direction;
    log(preset);
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut AutoTileMain");
        return;
    }
    focusMetaWindow = window;
    reset_window(window);
    let mind = window.get_monitor();
    getWorkAreaByMonitorIdx(mind);
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    let workArea = getWorkAreaByMonitor(monitor);
    let notFocusedwindows = getNotFocusedWindowsOfMonitor(monitor);
    if (Object.keys(notFocusedwindows).length === 0) {
        moveResizeWindowWithMargins(focusMetaWindow, workArea.x, workArea.y, workArea.width, workArea.height);
        return;
    }
    log("SETTINGS_MAIN_WINDOW_SIZES:" + mainWindowSizes);
    let ps_variants = mainWindowSizes;
    log("Main window sizes: " + ps_variants);
    // handle preset variants (if there are any)
    let variantCount = ps_variants.length;
    if (variantCount > 1) {
        if ((lastResizeInfo.lastCallTime.getTime() + gridSettings[SETTINGS_MAX_TIMEOUT]) > new Date().getTime() &&
            lastResizeInfo.presetName === preset.toString() &&
            lastResizeInfo.windowTitle == window.get_title()) {
            // within timeout (default: 2s), same preset & same window:
            // increase variant counter, but consider upper boundary
            lastResizeInfo.variantIndex = (lastResizeInfo.variantIndex + 1) % variantCount;
        }
        else {
            // timeout, new preset or new window:
            // update presetState.last_preset and reset variant counter
            lastResizeInfo.variantIndex = 0;
        }
    }
    else {
        lastResizeInfo.variantIndex = 0;
    }
    let mainRatio = ps_variants[lastResizeInfo.variantIndex];
    let mainWidth = workArea.width * mainRatio;
    let minorWidth = workArea.width - mainWidth;
    let mainX = 0;
    let minorX = 0;
    switch (tile_direction) {
        case 'left':
            mainX = 0;
            minorX = mainWidth;
            break;
        case 'right':
            mainX = minorWidth;
            minorX = 0;
            break;
    }
    moveResizeWindowWithMargins(focusMetaWindow, workArea.x + mainX, workArea.y, mainWidth, workArea.height);
    let winHeight = workArea.height / notFocusedwindows.length;
    let countWin = 0;
    log("AutoTileMain MonitorHeight: " + monitor.height + ":" + notFocusedwindows.length);
    for (let windowIdx in notFocusedwindows) {
        let metaWindow = notFocusedwindows[windowIdx].meta_window;
        let newOffset = workArea.y + (countWin * winHeight);
        reset_window(metaWindow);
        moveResizeWindowWithMargins(metaWindow, workArea.x + minorX, newOffset, minorWidth, winHeight);
        countWin++;
    }
    log("AutoTileMain done");
    lastResizeInfo.presetName = preset;
    lastResizeInfo.windowTitle = window.get_title();
    lastResizeInfo.lastCallTime = new Date();
    log("Resize preset last call: " + lastResizeInfo.lastCallTime);
}
class AutoTileTwoList extends ActionButton {
    constructor(grid) {
        const theme = `${getTheme()}${ACTION_CLASSES.TWO_LIST}`;
        super(grid, theme);
        this.classname = theme;
        log("AutoTileTwoList connect button-press-event");
        this.connect('button-press-event', () => this._onButtonPress());
    }
    _onButtonPress() {
        log("AutotileTwoList");
        autoTileNCols(2);
        log("AutoTileTwoList _onButtonPress Emitting signal resize-done");
        this.emit('resize-done');
        log("Autotile2 done");
    }
}
Signals.addSignalMethods(AutoTileTwoList.prototype);
function autoTileNCols(cols) {
    log("AutoTileNCols " + cols);
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut AutoTileNCols");
        return;
    }
    reset_window(window);
    let mind = window.get_monitor();
    getWorkAreaByMonitorIdx(mind);
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    let workArea = getWorkAreaByMonitor(monitor);
    if (!workArea) {
        return;
    }
    let windows = getNotFocusedWindowsOfMonitor(monitor);
    let nbWindowOnEachSide = Math.ceil((windows.length + (globalApp.isGridShowing() ? 1 : 0)) / cols);
    let winHeight = workArea.height / nbWindowOnEachSide;
    let countWin = 0;
    if (globalApp.isGridShowing()) {
        moveResizeWindowWithMargins(window, workArea.x + countWin % cols * workArea.width / cols, workArea.y + (Math.floor(countWin / cols) * winHeight), workArea.width / cols, winHeight);
        countWin++;
    }
    // todo make function
    for (let windowIdx in windows) {
        let metaWindow = windows[windowIdx].meta_window;
        reset_window(metaWindow);
        moveResizeWindowWithMargins(metaWindow, workArea.x + countWin % cols * workArea.width / cols, workArea.y + (Math.floor(countWin / cols) * winHeight), workArea.width / cols, winHeight);
        countWin++;
    }
    log("AutoTileNCols done");
}
function snapToNeighborsBind() {
    log("SnapToNeighbors keybind invoked");
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut SnapToNeighbors");
        return;
    }
    snapToNeighbors(window);
}
/**
 * GridSettingsButton is a GUI button that displays text like "8x8" and changes
 * the main grid size used for GUI elements and some presets.
 */
class GridSettingsButton {
    constructor(gridSize, active) {
        this.text = gridSize.toString();
        this.cols = gridSize.width;
        this.rows = gridSize.height;
        this.active = active;
        this.actor = new St.Button({
            style_class: `${theme}__preset-button`,
            reactive: true,
            can_focus: true,
            track_hover: true,
            label: this.text,
        });
        if (this.active) {
            this.activate();
        }
        this.actor.connect('button-press-event', () => this._onButtonPress());
    }
    deactivate() {
        log("Deactivating GridSettingsButton ${cols}:${rows}");
        this.active = false;
        this.actor.remove_style_pseudo_class('activate');
    }
    activate() {
        log("Activating GridSettingsButton ${cols}:${rows}");
        this.active = true;
        this.actor.add_style_pseudo_class('activate');
    }
    _onButtonPress() {
        nbCols = this.cols;
        nbRows = this.rows;
        globalApp.refreshGrids();
    }
}
const GRID_WIDGET_BORDER_SIZE = 1;
class Grid {
    constructor(gridWidget, monitor_idx, monitor, title, cols, rows) {
        this.gridWidget = gridWidget;
        this.tableWidth = 320;
        this.borderwidth = 2;
        this.elements = [];
        this.moveResizeImpl = null;
        let workArea = getWorkArea(monitor, monitor_idx);
        this.tableHeight = (this.tableWidth / workArea.width) * workArea.height;
        this.actor = new St.BoxLayout({
            vertical: true,
            style_class: theme,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        log(`created actor for monitor ${monitor_idx}: ${this.actor} with cols=${cols}, rows=${rows}`);
        log("Grid connect enter-event leave-envent BEGIN");
        // TODO: disconnect these events on teardown.
        this.actor.connect('enter-event', () => this._onMouseEnter());
        this.actor.connect('leave-event', () => this._onMouseLeave());
        log("Grid connect enter-event leave-envent FINISH");
        this.animation_time = gridSettings[SETTINGS_ANIMATION] ? 0.3 : 0;
        this.topbar = new TopBar(title);
        this.bottombarContainer = new St.Bin({
            style_class: `${theme}__preset-container`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.bottomBarTableLayout = new Clutter.GridLayout();
        this.bottombar = new St.Widget({
            style_class: `${theme}__preset`,
            can_focus: true,
            track_hover: true,
            reactive: true,
            width: this.tableWidth - 20,
            height: 36,
            layout_manager: this.bottomBarTableLayout
        });
        this.bottomBarTableLayout.set_row_homogeneous(true);
        this.bottomBarTableLayout.set_column_homogeneous(true);
        this.bottombarContainer.add_actor(this.bottombar);
        this.veryBottomBarContainer = new St.Bin({
            style_class: `${theme}__action-container`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.veryBottomBarTableLayout = new Clutter.GridLayout();
        this.veryBottomBar = new St.Widget({
            style_class: `${theme}__action`,
            can_focus: true,
            track_hover: true,
            reactive: true,
            width: this.tableWidth - 20,
            height: 36,
            layout_manager: this.veryBottomBarTableLayout,
        });
        this.bottomBarTableLayout.set_row_homogeneous(true);
        this.veryBottomBarTableLayout.set_column_homogeneous(true);
        this.veryBottomBarContainer.add_actor(this.veryBottomBar);
        let rowNum = 0;
        let colNum = 0;
        let maxPerRow = 4;
        this.rows = rows;
        this.cols = cols;
        this.gridSettingsButtons = [];
        for (let gridSize of gridSettings[SETTINGS_GRID_SIZES]) {
            if (colNum >= maxPerRow) {
                colNum = 0;
                rowNum += 1;
            }
            const isActiveGrid = this.cols == gridSize.width && this.rows == gridSize.height;
            const button = new GridSettingsButton(gridSize, isActiveGrid);
            this.bottomBarTableLayout.attach(button.actor, colNum, rowNum, 1, 1);
            button.actor.connect('notify::hover', () => this._onSettingsButton());
            colNum++;
            this.gridSettingsButtons.push(button);
        }
        this.bottombar.height *= (rowNum + 1);
        this.tableContainer = new St.Bin({
            style_class: `${theme}__tile-container`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.tableLayoutManager = new Clutter.GridLayout();
        this.table = new St.Widget({
            style_class: `${theme}__tile-table`,
            can_focus: true,
            track_hover: true,
            reactive: true,
            height: this.tableHeight,
            width: this.tableWidth - GRID_WIDGET_BORDER_SIZE * 2,
            layout_manager: this.tableLayoutManager,
        });
        this.tableLayoutManager.set_row_homogeneous(true);
        this.tableLayoutManager.set_column_homogeneous(true);
        this.tableContainer.add_actor(this.table);
        this.actor.add_child(this.topbar.actor);
        this.actor.add_child(this.tableContainer);
        this.actor.add_child(this.bottombarContainer);
        this.actor.add_child(this.veryBottomBarContainer);
        this.monitor = monitor;
        this.monitor_idx = monitor_idx;
        this.title = title;
        this.isEntered = false;
        const toggleSettingListener = new ToggleSettingsButtonListener();
        let toggle = new ToggleSettingsButton("animation", SETTINGS_ANIMATION);
        this.veryBottomBarTableLayout.attach(toggle.actor, 0, 0, 1, 1);
        toggleSettingListener.addActor(toggle);
        toggle = new ToggleSettingsButton("auto-close", SETTINGS_AUTO_CLOSE);
        this.veryBottomBarTableLayout.attach(toggle.actor, 1, 0, 1, 1);
        toggleSettingListener.addActor(toggle);
        let action = new AutoTileMainAndList(this);
        this.veryBottomBarTableLayout.attach(action.actor, 2, 0, 1, 1);
        action.connect('resize-done', this._onResize.bind(this));
        action = new AutoTileTwoList(this);
        this.veryBottomBarTableLayout.attach(action.actor, 3, 0, 1, 1);
        action.connect('resize-done', this._onResize.bind(this));
        this.x = 0;
        this.y = 0;
        this.interceptHide = false;
        this._displayElements();
        this.normalScaleY = this.actor.scale_y;
        this.normalScaleX = this.actor.scale_x;
    }
    state() {
        return `grid with actor ${this.actor}`;
    }
    _displayElements() {
        if (this.monitor === null) {
            return;
        }
        log("Grid _displayElements " + this.cols + ":" + this.rows);
        this.elements = new Array();
        let width = (this.tableWidth / this.cols); // - 2*this.borderwidth;
        let height = (this.tableHeight / this.rows); // - 2*this.borderwidth;
        const delegate = new GridElementDelegate(this, this.gridWidget);
        this.elementsDelegate = delegate;
        this.elementsDelegate.connect('resize-done', (actor, event) => this._onResize());
        for (let r = 0; r < this.rows; r++) {
            for (let c = 0; c < this.cols; c++) {
                if (c == 0) {
                    this.elements[r] = new Array();
                }
                const element = new GridElement(this.elementsDelegate, this.monitor, width, height, c, r);
                this.elements[r][c] = element;
                this.tableLayoutManager.attach(element.actor, c, r, 1, 1);
                element.show();
            }
        }
    }
    getElement(row, col) {
        return this.elements[row][col] || null;
    }
    forceGridElementDelegate(x, y, w, h) {
        var _a;
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.forceArea(this.elements[y][x], this.elements[h][w]);
    }
    refresh() {
        log("Grid.refresh from " + this.cols + ":" + this.rows + " to " + nbCols + ":" + nbRows);
        this.elementsDelegate._resetGrid();
        for (let r = 0; r < this.rows; r++) {
            for (let c = 0; c < this.cols; c++) {
                this.elements[r][c]._disconnect();
            }
        }
        this.table.destroy_all_children();
        this.cols = nbCols;
        this.rows = nbRows;
        this._displayElements();
        this._clearMoveResizeState();
        this._updateGridSizeButtons();
    }
    set_position(x, y) {
        this.x = x;
        this.y = y;
        this.actor.set_position(x, y);
    }
    show() {
        this.interceptHide = true;
        this.elementsDelegate.reset();
        let time = (gridSettings[SETTINGS_ANIMATION]) ? 0.3 : 0;
        Main.uiGroup.set_child_above_sibling(this.actor, null);
        Main.layoutManager.removeChrome(this.actor);
        Main.layoutManager.addChrome(this.actor);
        //this.actor.y = 0 ;
        if (time > 0) {
            this.actor.scale_y = 0;
            this.actor.scale_x = 0;
            this.actor.ease({
                time: this.animation_time,
                opacity: 255,
                visible: true,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                scale_x: this.normalScaleX,
                scale_y: this.normalScaleY,
                onComplete: this._onShowComplete
            });
        }
        else {
            this.actor.scale_x = this.normalScaleX;
            this.actor.scale_y = this.normalScaleY;
            this.actor.opacity = 255;
            this.actor.visible = true;
        }
        this.interceptHide = false;
    }
    hide(immediate) {
        var _a;
        log(`Grid hide immediate=${immediate}`);
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
        let time = (gridSettings[SETTINGS_ANIMATION]) ? 0.3 : 0;
        if (!immediate && time > 0) {
            this.actor.ease({
                time: this.animation_time,
                opacity: 0,
                visible: false,
                scale_x: 0,
                scale_y: 0,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                onComplete: this._onHideComplete
            });
        }
        else {
            this.actor.opacity = 0;
            this.actor.visible = false;
            //this.actor.y = 0;
            this.actor.scale_x = 0;
            this.actor.scale_y = 0;
        }
        this._clearMoveResizeState();
    }
    setInitialSelection(focusMetaWindow) {
        var _a, _b, _c, _d;
        // The window being focused is part of the monitor that this Grid is
        // responsible for manipulating.
        if (focusMetaWindow.get_monitor() !== this.monitor_idx) {
            return;
        }
        if (!this.monitor) {
            return;
        }
        const monitor = this.monitor;
        const workArea = getWorkArea(this.monitor, this.monitor_idx);
        let wx = focusMetaWindow.get_frame_rect().x;
        let wy = focusMetaWindow.get_frame_rect().y;
        let wwidth = focusMetaWindow.get_frame_rect().width;
        let wheight = focusMetaWindow.get_frame_rect().height;
        const grid = this;
        const delegate = grid.elementsDelegate;
        log("Set initial selection");
        log("Focus window position x " + wx + " y " + wy + " width " + wwidth + " height " + wheight);
        log("Focus monitor position x " + monitor.x + " y " + monitor.y + " width " + monitor.width + " height " + monitor.height);
        log("Workarea position x " + workArea.x + " y " + workArea.y + " width " + workArea.width + " height " + workArea.height);
        let wax = Math.max(wx - workArea.x, 0);
        let way = Math.max(wy - workArea.y, 0);
        let grid_element_width = workArea.width / nbCols;
        let grid_element_height = workArea.height / nbRows;
        log("width " + grid_element_width + " height " + grid_element_height);
        let lux = Math.min(Math.max(Math.round(wax / grid_element_width), 0), nbCols - 1);
        log("wx " + (wx - workArea.x) + " el_width " + grid_element_width + " max " + (nbCols - 1) + " res " + lux);
        let luy = Math.min(Math.max(Math.round(way / grid_element_height), 0), grid.rows - 1);
        log("wy " + (wy - workArea.y) + " el_height " + grid_element_height + " max " + (nbRows - 1) + " res " + luy);
        let rdx = Math.min(Math.max(Math.round((wax + wwidth) / grid_element_width) - 1, lux), grid.cols - 1);
        log("wx + wwidth " + (wx + wwidth - workArea.x - 1) + " el_width " + grid_element_width + " max " + (nbCols - 1) + " res " + rdx);
        let rdy = Math.min(Math.max(Math.round((way + wheight) / grid_element_height) - 1, luy), grid.rows - 1);
        log("wy + wheight " + (wy + wheight - workArea.y - 1) + " el_height " + grid_element_height + " max " + (nbRows - 1) + " res " + rdy);
        log("Initial tile selection is " + lux + ":" + luy + " - " + rdx + ":" + rdy);
        grid.forceGridElementDelegate(lux, luy, rdx, rdy);
        grid.elements[luy][lux]._onButtonPress();
        grid.elements[rdy][rdx]._onHoverChanged();
        const cX = (_a = delegate === null || delegate === void 0 ? void 0 : delegate.currentElement) === null || _a === void 0 ? void 0 : _a.coordx;
        const cY = (_b = delegate === null || delegate === void 0 ? void 0 : delegate.currentElement) === null || _b === void 0 ? void 0 : _b.coordy;
        const fX = (_c = delegate === null || delegate === void 0 ? void 0 : delegate.first) === null || _c === void 0 ? void 0 : _c.coordx;
        const fY = (_d = delegate === null || delegate === void 0 ? void 0 : delegate.first) === null || _d === void 0 ? void 0 : _d.coordy;
        log("After initial selection first fX " + fX + " fY " + fY + " current cX " + cX + " cY " + cY);
    }
    moveResize(window, type, key) {
        const delegate = this.elementsDelegate;
        if (!delegate) {
            return false;
        }
        if (!delegate.currentElement) {
            log("Key event while no mouse activation - set current and second element");
            this.setInitialSelection(window);
        }
        if (!delegate.currentElement) {
            log("bug: currentElement must be set in moveResize");
            return;
        }
        if (!delegate.first) {
            log("currentElement is there but no first yet");
            delegate.currentElement._onButtonPress();
        }
        if (!delegate.first) {
            log("bug: first must be set in moveResize");
            return;
        }
        if (this.moveResizeImpl == null) {
            this.moveResizeImpl = new MoveResizeImpl(delegate.currentElement, delegate.first);
        }
        return this.moveResizeImpl.moveResize(this, delegate, type, key);
    }
    _clearMoveResizeState() {
        log("Clear moveResize state");
        this.moveResizeImpl = null;
    }
    _onHideComplete() {
    }
    _onShowComplete() {
    }
    _onResize() {
        log("resize-done");
        globalApp.updateRegions();
        if (gridSettings[SETTINGS_AUTO_CLOSE]) {
            log("Emitting hide-tiling");
            this.emit('hide-tiling');
        }
        this._clearMoveResizeState();
    }
    _onMouseEnter() {
        var _a;
        log("Grid: onMouseEnter");
        if (!this.isEntered) {
            (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
            this.isEntered = true;
        }
    }
    _onMouseLeave() {
        var _a;
        log("Grid: onMouseLeave");
        let [x, y, mask] = global.get_pointer();
        if (this.elementsDelegate && (x <= this.actor.x || x >= (this.actor.x + this.actor.width)) || (y <= this.actor.y || y >= (this.actor.y + this.actor.height))) {
            this.isEntered = false;
            (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
            globalApp.refreshGrids();
        }
    }
    _onSettingsButton() {
        var _a;
        log("_onSettingsButton");
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
    }
    _destroy() {
        var _a;
        log("Grid _destroy");
        for (let r in this.elements) {
            for (let c in this.elements[r]) {
                this.elements[r][c]._destroy();
            }
        }
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a._destroy();
        this.topbar.destroy();
        this.monitor = null;
        this.rows = 0;
        this.title = "";
        this.cols = 0;
        log("Disconnect hide-tiling");
        this.disconnect(this.connectHideTiling);
    }
    _updateGridSizeButtons() {
        for (let button of this.gridSettingsButtons) {
            if (this.cols == button.cols && this.rows == button.rows) {
                button.activate();
            }
            else {
                button.deactivate();
            }
        }
    }
    // Methods replaced by Signals.addSignalMethods.
    connect(name, callback) { return -1; }
    disconnect(id) { }
    emit(name, ...args) { }
}
class MoveResizeImpl {
    constructor(current, first) {
        const cX = current.coordx;
        const cY = current.coordy;
        const fX = first.coordx;
        const fY = first.coordy;
        this.vW = Math.abs(cX - fX) + 1;
        this.vH = Math.abs(cY - fY) + 1;
        this.vX = Math.min(cX, fX);
        this.vY = Math.min(cY, fY);
    }
    moveResize(grid, delegate, type, key) {
        var _a;
        const cols = grid.cols;
        const rows = grid.rows;
        log(`Before move/resize vX = ${this.vX}, vY = ${this.vY}, vW = ${this.vW}, vH = ${this.vH}`);
        if (type === 'move') {
            switch (key) {
                case 'right':
                    if (this.vX < cols - 1) {
                        this.vX += 1;
                    }
                    break;
                case 'left':
                    if (0 < this.vX + this.vW - 1) {
                        this.vX -= 1;
                    }
                    break;
                case 'up':
                    if (0 < this.vY + this.vH - 1) {
                        this.vY -= 1;
                    }
                    break;
                case 'down':
                    if (this.vY < rows - 1) {
                        this.vY += 1;
                    }
                    break;
            }
        }
        else if (type == "resize") {
            switch (key) {
                case 'right':
                    if (this.vW < cols) {
                        this.vW += 1;
                    }
                    break;
                case 'left':
                    if (this.vW > 1) {
                        this.vW -= 1;
                    }
                    break;
                case 'up':
                    if (this.vH > 1) {
                        this.vH -= 1;
                    }
                    break;
                case 'down':
                    if (this.vH < rows) {
                        this.vH += 1;
                    }
                    break;
            }
        }
        else if (type == "contract") {
            switch (key) {
                case 'left':
                    // Contract left edge of current window right one column
                    if (this.vX < cols - 1 && this.vW > 1) {
                        this.vX += 1;
                        this.vW -= 1;
                    }
                    break;
                case 'right':
                    // Contract right edge of current window left one column
                    if (0 < this.vX + this.vW - 1 && this.vW > 1) {
                        this.vW -= 1;
                    }
                    break;
                case 'top':
                    // Contract top edge of current window down one row
                    if (this.vY < rows - 1 && this.vH > 1) {
                        this.vY += 1;
                        this.vH -= 1;
                    }
                    break;
                case 'bottom':
                    // Contract bottom edge of current window up one row
                    if (0 < this.vY + this.vH - 1 && this.vH > 1) {
                        this.vH -= 1;
                    }
                    break;
            }
        }
        else if (type == "expand") {
            switch (key) {
                case 'right':
                    if (this.vW < cols) {
                        this.vW += 1;
                    }
                    break;
                case 'left':
                    if (this.vW < cols) {
                        this.vW += 1;
                        this.vX -= 1;
                    }
                    break;
                case 'top':
                    if (this.vH < rows) {
                        this.vH += 1;
                        this.vY -= 1;
                    }
                    break;
                case 'bottom':
                    if (this.vH < rows) {
                        this.vH += 1;
                    }
                    break;
            }
        }
        var tFX = Math.max(0, this.vX);
        var tCX = Math.min(this.vX + this.vW - 1, cols - 1);
        var tFY = Math.max(0, this.vY);
        var tCY = Math.min(this.vY + this.vH - 1, rows - 1);
        delegate.first = grid.getElement(tFY, tFX);
        (_a = grid.getElement(tCY, tCX)) === null || _a === void 0 ? void 0 : _a._onHoverChanged();
        return true;
    }
}
Signals.addSignalMethods(Grid.prototype);
class GridElementDelegate {
    constructor(grid, gridWidget) {
        this.grid = grid;
        this.gridWidget = gridWidget;
        // The first element clicked in the rectangular selection.
        this.first = null;
        //
        this.currentElement = null;
        // Elements that are in a highlighted state in the UI. The highlighting
        // happens when the grid rectangle selcted includes a particular grid
        // element.
        this.activatedElements = [];
        this.activated = false;
    }
    allSelected() {
        return (this.activatedElements.length === (nbCols * nbRows));
    }
    onButtonPress(gridElement) {
        log("GridElementDelegate _onButtonPress " + gridElement.coordx + ":" + gridElement.coordy);
        //this._logActiveActors("GridElementDelegate _onButtonPress active actors");
        if (!this.currentElement) {
            this.currentElement = gridElement;
        }
        if (!this.activated) {
            log("GridElementDelegate first activation");
            this.activated = true;
            gridElement.active = true;
            this.activatedElements = [gridElement];
            this.first = gridElement;
            return;
        }
        log("GridElementDelegate resize");
        //Check this.activatedActors if equals to nbCols * nbRows
        //before doing anything with the window it must be unmaximized
        //if so move the window then maximize instead of change size
        //if not move the window and change size
        if (!focusMetaWindow) {
            return;
        }
        reset_window(focusMetaWindow);
        if (this.first) {
            const computedSize = this._computeAreaPositionSize(this.first, gridElement);
            if (!computedSize) {
                return;
            }
            const [areaX, areaY, areaWidth, areaHeight] = computedSize;
            if (this.allSelected() && gridSettings[SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED] === false) {
                move_maximize_window(focusMetaWindow, areaX, areaY);
            }
            else {
                moveResizeWindowWithMargins(focusMetaWindow, areaX, areaY, areaWidth, areaHeight);
            }
        }
        this._resizeDone();
    }
    _resizeDone() {
        log("resizeDone, emitting signal resize-done");
        this.emit('resize-done');
    }
    reset() {
        this._resetGrid();
        this.activated = false;
        this.first = null;
        this.currentElement = null;
    }
    _resetGrid() {
        this._hideArea();
        if (this.currentElement) {
            this.currentElement._deactivate();
        }
        for (let act of this.activatedElements) {
            act._deactivate();
        }
        this.activatedElements = new Array();
    }
    _getVarFromGridElement(fromGridElement, toGridElement) {
        let minX = Math.min(fromGridElement.coordx, toGridElement.coordx);
        let maxX = Math.max(fromGridElement.coordx, toGridElement.coordx);
        let minY = Math.min(fromGridElement.coordy, toGridElement.coordy);
        let maxY = Math.max(fromGridElement.coordy, toGridElement.coordy);
        return [minX, maxX, minY, maxY];
    }
    refreshGrid(fromGridElement, toGridElement) {
        this._resetGrid();
        let [minX, maxX, minY, maxY] = this._getVarFromGridElement(fromGridElement, toGridElement);
        if (!fromGridElement.monitor) {
            return;
        }
        for (let r = minY; r <= maxY; r++) {
            for (let c = minX; c <= maxX; c++) {
                const highlightedElement = this.grid.getElement(r, c);
                if (!highlightedElement) {
                    log(`bug: GridElementDelegate refreshGrid is out of sync with Grid: trying to highlight element (row=${r}, col=${c}) and getElement() returned null.`);
                    continue;
                }
                highlightedElement._activate();
                this.activatedElements.push(highlightedElement);
            }
        }
        this._displayArea(fromGridElement, toGridElement);
    }
    _computeAreaPositionSize(fromGridElement, toGridElement) {
        let [minX, maxX, minY, maxY] = this._getVarFromGridElement(fromGridElement, toGridElement);
        let monitor = fromGridElement.monitor;
        const workArea = getWorkAreaByMonitor(monitor);
        if (!workArea) {
            return null;
        }
        let areaWidth = Math.round((workArea.width / nbCols) * ((maxX - minX) + 1));
        let areaHeight = Math.round((workArea.height / nbRows) * ((maxY - minY) + 1));
        let areaX = workArea.x + Math.round((minX * (workArea.width / nbCols)));
        let areaY = workArea.y + Math.round((minY * (workArea.height / nbRows)));
        return [areaX, areaY, areaWidth, areaHeight];
    }
    forceArea(fromGridElement, toGridElement) {
        const computedSize = this._computeAreaPositionSize(fromGridElement, toGridElement);
        if (!computedSize) {
            return;
        }
        const [areaX, areaY, areaWidth, areaHeight] = computedSize;
        this.gridWidget.width = areaWidth;
        this.gridWidget.height = areaHeight;
        this.gridWidget.x = areaX;
        this.gridWidget.y = areaY;
    }
    _displayArea(fromGridElement, toGridElement) {
        const computedSize = this._computeAreaPositionSize(fromGridElement, toGridElement);
        if (!computedSize) {
            return;
        }
        const [areaX, areaY, areaWidth, areaHeight] = computedSize;
        this.gridWidget.add_style_pseudo_class('activate');
        if (gridSettings[SETTINGS_ANIMATION]) {
            this.gridWidget.ease({
                time: 0.2,
                x: areaX,
                y: areaY,
                width: areaWidth,
                height: areaHeight,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD
            });
        }
        else {
            this.gridWidget.width = areaWidth;
            this.gridWidget.height = areaHeight;
            this.gridWidget.x = areaX;
            this.gridWidget.y = areaY;
        }
    }
    _hideArea() {
        this.gridWidget.remove_style_pseudo_class('activate');
    }
    _onHoverChanged(gridElement) {
        log("GridElementDelegate _onHoverChange " + gridElement.coordx + ":" + gridElement.coordy);
        if (this.activated) {
            log("GridElementDelegate _onHoverChange/not active: " + gridElement.coordx + ":" + gridElement.coordy);
            this.refreshGrid(this.first, gridElement);
            this.currentElement = gridElement;
        }
        else if (!this.currentElement || gridElement.id != this.currentElement.id) {
            log("GridElementDelegate _onHoverChange/active: " + gridElement.coordx + ":" + gridElement.coordy);
            if (this.currentElement) {
                this.currentElement._deactivate();
            }
            this.currentElement = gridElement;
            this.currentElement._activate();
            this._displayArea(gridElement, gridElement);
        }
        else {
            log("GridElementDelegate _onHoverChange/else: " + gridElement.coordx + ":" + gridElement.coordy);
        }
    }
    _destroy() {
        this.activated = false;
        this.first = null;
        this.currentElement = null;
        this.activatedElements = [];
    }
    // Methods replaced by Signals.addSignalMethods.
    connect(name, callback) { return -1; }
    disconnect(id) { }
    emit(name, ...args) { }
}
Signals.addSignalMethods(GridElementDelegate.prototype);
class GridElement {
    constructor(delegate, monitor, width, height, coordx, coordy) {
        this.delegate = delegate;
        this.monitor = monitor;
        this.width = width;
        this.height = height;
        this.coordx = coordx;
        this.coordy = coordy;
        this.actor = new St.Button({
            style_class: `${theme}__tile-table-item`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.actor.visible = false;
        this.actor.opacity = 0;
        this.id = getMonitorKey(monitor) + "-" + coordx + ":" + coordy;
        this.actor.connect('button-press-event', this._onButtonPress.bind(this));
        this.hoverConnect = this.actor.connect('notify::hover', () => this._onHoverChanged());
        this.active = false;
    }
    state() {
        return `id: ${this.id}; visible: ${this.actor.visible}`;
    }
    show() {
        this.actor.opacity = 255;
        this.actor.visible = true;
    }
    hide() {
        this.actor.opacity = 0;
        this.actor.visible = false;
    }
    _onButtonPress() {
        this.delegate.onButtonPress(this);
    }
    _onHoverChanged() {
        this.delegate._onHoverChanged(this);
    }
    _activate() {
        if (!this.active) {
            this.actor.add_style_pseudo_class('activate');
            this.active = true;
        }
    }
    _deactivate() {
        if (this.active) {
            this.actor.remove_style_pseudo_class('activate');
            this.active = false;
        }
    }
    // This logic should probably go into disable().
    // _clean() {
    //     Main.uiGroup.remove_actor(this.gridWidget);
    // }
    _disconnect() {
        this.actor.disconnect(this.hoverConnect);
    }
    _destroy() {
        this.active = false;
    }
}
function getTheme() {
    let themeName = settings.get_string(SETTINGS_THEME) || 'Default';
    themeName = themeName.toLowerCase().replace(/[^A-Za-z0-9]/g, '-');
    const theme = `gtile-${themeName}`;
    log("Theme changed to the" + theme);
    return theme;
}

registerExtension(function () { }, enable, disable);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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
