# -*- coding: binary -*-
require 'rex/proto/kerberos/pac/krb5_pac'

module Msf
  class Exploit
    class Remote
      module Kerberos
        module Client
          module Pac

            # Builds a kerberos PA-PAC-REQUEST pre authenticated structure
            #
            # @param opts [Hash{Symbol => Boolean}]
            # @option opts [Boolean] :pac_request_value
            # @return [Rex::Proto::Kerberos::Model::Field::PreAuthDataEntry]
            # @see Rex::Proto::Kerberos::Model::PreAuthPacRequest
            # @see Rex::Proto::Kerberos::Model::PreAuthDataEntry
            def build_pa_pac_request(opts = {})
              value = opts[:pac_request_value] || false
              pac_request = Rex::Proto::Kerberos::Model::PreAuthPacRequest.new(value: value)
              pa_pac_request = Rex::Proto::Kerberos::Model::PreAuthDataEntry.new(
                type: Rex::Proto::Kerberos::Model::PreAuthType::PA_PAC_REQUEST,
                value: pac_request.encode
              )

              pa_pac_request
            end

            # Builds a kerberos PACTYPE structure
            #
            # @param opts [Hash{Symbol => <String, Integer, Array, Time>}]
            # @option opts [String] :client_name
            # @option opts [Integer] :user_id the user SID Ex: 1000
            # @option opts [Integer] :group_id Ex: 513 for 'Domain Users'
            # @option opts [Array<Integer>] :group_ids
            # @option opts [Array<String>] :extra_sids An array of extra sids, Ex: `['S-1-5-etc-etc-519']`
            # @option opts [String] :realm
            # @option opts [String] :domain_id the domain SID Ex: S-1-5-21-1755879683-3641577184-3486455962
            # @option opts [Time] :logon_time
            # @option opts[String] :checksum_enc_key Encryption key for calculating the checksum
            # @return [Rex::Proto::Kerberos::Pac::Krb5Pac]
            # @see Rex::Proto::Kerberos::Pac::Krb5PacLogonInfo
            # @see Rex::Proto::Kerberos::Pac::Krb5PacClientInfo
            # @see Rex::Proto::Kerberos::Pac::Krb5PacServerChecksum
            # @see Rex::Proto::Kerberos::Pac::Krb5PacPrivSvrChecksum
            # @see Rex::Proto::Kerberos::Pac::Krb5Pac
            def build_pac(opts = {})
              user_name = opts[:client_name] || ''
              user_id = opts[:user_id] || Rex::Proto::Kerberos::Pac::DEFAULT_ADMIN_RID
              primary_group_id = opts[:group_id] || Rex::Proto::Kerberos::Pac::DOMAIN_USERS
              group_ids = opts[:group_ids] || [Rex::Proto::Kerberos::Pac::DOMAIN_USERS]
              extra_sids = opts[:extra_sids] || []
              domain_name = opts[:realm] || ''
              domain_id = opts[:domain_id] || Rex::Proto::Kerberos::Pac::NT_AUTHORITY_SID
              logon_time = opts[:logon_time] || Time.now
              checksum_type = opts[:checksum_type] || Rex::Proto::Kerberos::Crypto::Checksum::RSA_MD5
              ticket_checksum = opts[:ticket_checksum] || nil

              validation_info = Rex::Proto::Kerberos::Pac::Krb5ValidationInfo.new(
                logon_time: logon_time,
                effective_name: user_name,
                user_id: user_id,
                primary_group_id: primary_group_id,
                logon_domain_name: domain_name,
                logon_domain_id: domain_id,
                full_name: '',
                logon_script: '',
                profile_path: '',
                home_directory: '',
                home_directory_drive: '',
                logon_server: ''
              )
              validation_info.group_ids = group_ids
              if extra_sids && extra_sids.length > 0
                validation_info.extra_sids = extra_sids.map do |sid|
                  { sid: sid, attributes: Rex::Proto::Kerberos::Pac::SE_GROUP_ALL }
                end
              end

              logon_info = Rex::Proto::Kerberos::Pac::Krb5LogonInformation.new(
                data: validation_info
              )

              client_info = Rex::Proto::Kerberos::Pac::Krb5ClientInfo.new(
                client_id: logon_time,
                name: user_name
              )

              server_checksum = Rex::Proto::Kerberos::Pac::Krb5PacServerChecksum.new(
                signature_type: checksum_type
              )

              priv_srv_checksum = Rex::Proto::Kerberos::Pac::Krb5PacPrivServerChecksum.new(
                signature_type: checksum_type
              )

              pac_elements = [
                logon_info,
                client_info,
                server_checksum,
                priv_srv_checksum
              ]
              pac_elements << ticket_checksum unless ticket_checksum.nil?

              pac_type = Rex::Proto::Kerberos::Pac::Krb5Pac.new
              pac_type.assign(pac_elements: pac_elements)
              pac_type.sign!(service_key: opts[:checksum_enc_key])
              pac_type
            end

            # Builds an kerberos AuthorizationData structure containing a PACTYPE
            #
            # @param opts [Hash{Symbol => Rex::Proto::Kerberos::Pac::Type}]
            # @option opts [Rex::Proto::Kerberos::Pac::Type] :pac
            # @return [Rex::Proto::Kerberos::Model::AuthorizationData]
            # @see Rex::Proto::Kerberos::Model::AuthorizationData
            def build_pac_authorization_data(opts = {})
              pac = opts[:pac] || build_pac(opts)

              pac_auth_data = Rex::Proto::Kerberos::Model::AuthorizationData.new(
                elements: [{ type: Rex::Proto::Kerberos::Pac::AD_WIN2K_PAC, data: pac.to_binary_s}]
              )
              authorization_data = Rex::Proto::Kerberos::Model::AuthorizationData.new(
                elements: [{ type: Rex::Proto::Kerberos::Model::AuthorizationDataType::AD_IF_RELEVANT, data: pac_auth_data.encode }]
              )

              authorization_data
            end

            def build_empty_auth_data
              pac_auth_data = Rex::Proto::Kerberos::Model::AuthorizationData.new(
                elements: [{ type: Rex::Proto::Kerberos::Pac::AD_WIN2K_PAC, data: "\x00" }]
              )
              Rex::Proto::Kerberos::Model::AuthorizationData.new(
                elements: [{ type: Rex::Proto::Kerberos::Model::AuthorizationDataType::AD_IF_RELEVANT, data: pac_auth_data.encode }]
              )
            end
          end
        end
      end
    end
  end
end
