<?php

declare(strict_types=1);
/**
 * @copyright Copyright (c) 2019 Joas Schilling <coding@schilljs.com>
 *
 * @license GNU AGPL version 3 or any later version
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

namespace OCA\Talk\Command\Command;

use OC\Core\Command\Base;
use OCA\Talk\Service\CommandService;
use OCP\AppFramework\Db\DoesNotExistException;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class Update extends Base {
	use TRenderCommand;

	private CommandService $service;

	public function __construct(CommandService $service) {
		parent::__construct();
		$this->service = $service;
	}

	protected function configure(): void {
		parent::configure();
		$this
			->setName('talk:command:update')
			->setDescription('Add a new command')
			->addArgument(
				'command-id',
				InputArgument::REQUIRED
			)
			->addArgument(
				'cmd',
				InputArgument::REQUIRED,
				'The command as used in the chat "/help" => "help"'
			)
			->addArgument(
				'name',
				InputArgument::REQUIRED,
				'Name of the user posting the response'
			)
			->addArgument(
				'script',
				InputArgument::REQUIRED,
				'Script to execute (Must be using absolute paths only)'
			)
			->addArgument(
				'response',
				InputArgument::REQUIRED,
				'Who should see the response: 0 - No one, 1 - User, 2 - All'
			)
			->addArgument(
				'enabled',
				InputArgument::REQUIRED,
				'Who can use this command: 0 - Disabled, 1 - Moderators, 2 - Users, 3 - Guests'
			)
		;
	}

	protected function execute(InputInterface $input, OutputInterface $output): int {
		$id = (int) $input->getArgument('command-id');
		$cmd = $input->getArgument('cmd');
		$name = $input->getArgument('name');
		$script = $input->getArgument('script');
		$response = (int) $input->getArgument('response');
		$enabled = (int) $input->getArgument('enabled');

		try {
			$command = $this->service->update($id, $cmd, $name, $script, $response, $enabled);
		} catch (DoesNotExistException $e) {
			$output->writeln('<error>The given command ID does not exist</error>');
			return 1;
		} catch (\InvalidArgumentException $e) {
			switch ($e->getCode()) {
				case 0:
					$output->writeln('<error>The help command and predefined commands cannot be updated</error>');
					break;
				case 1:
					$output->writeln('<error>The command already exists or is invalid</error>');
					break;
				case 2:
					$output->writeln('<error>The name is invalid</error>');
					break;
				case 3:
					$output->writeln('<error>The script is invalid</error>');
					break;
				case 4:
					$output->writeln('<error>The response value is invalid</error>');
					break;
				case 5:
					$output->writeln('<error>The enabled value is invalid</error>');
					break;
				default:
					$output->writeln('<error>The command could not be updated</error>');
					break;
			}
			return 2;
		}


		$output->writeln('<info>Command updated</info>');
		$output->writeln('');
		$this->renderCommands($input, $output, [$command]);
		return 0;
	}
}
