/*!
 * OpenUI5
 * (c) Copyright 2009-2022 SAP SE or an SAP affiliate company.
 * Licensed under the Apache License, Version 2.0 - see LICENSE.txt.
 */
sap.ui.define([
	"sap/m/table/ItemBase"
], function(
	ItemBase
) {
	"use strict";

	/**
	 * Constructor for a new Item.
	 *
	 * @param {string} [sId] ID for the new control, generated automatically if no ID is given
	 * @param {object} [mSettings] Initial settings for the new control
	 *
	 * @class
	 * This control serves as a menu item for the sap.m.table.ColumnMenu.
	 * It can be used to specify control- and application-specific items.
	 *
	 * @extends sap.m.table.ItemBase
	 *
	 * @author SAP SE
	 * @version 1.98.0
	 *
	 * @private
	 * @experimental
	 *
	 * @alias sap.m.table.Item
	 */
	var Item = ItemBase.extend("sap.m.table.Item", {
		metadata: {
			library: "sap.m",
			properties: {
				/**
				 * Defines the label, which should be used for the buttons.
				 */
				label: {type: "string"},
				/**
				 * Defines the icon for the menu item.
				 */
				icon: {type: "sap.ui.core.URI"},
				/**
				 * Defines whether the reset button should be shown, when navigating to the item.
				 */
				showResetButton: {type: "boolean", defaultValue: true},
				/**
				 * Defines whether the reset button should be enabled, when navigating to the item.
				 */
				resetButtonEnabled: {type: "boolean", defaultValue: true},
				/**
				 * Defines whether the confirm button should be shown, when navigating to the item.
				 */
				showConfirmButton: {type: "boolean", defaultValue: true},
				/**
				 * Defines whether the cancel button should be shown, when navigating to the item.
				 */
				showCancelButton: {type: "boolean", defaultValue: true}
			},
			aggregations: {
				/**
				 * Defines the content, which should be shown when navigating to the item.
				 */
				content: {type: "sap.ui.core.Control", multiple: false}
			},
			events: {
				/**
				 * This event will be fired, when the reset button was pressed.
				 */
				reset: {},
				/**
				 * This event will be fired, when the confirm button was pressed.
				 */
				confirm: {},
				/**
				 * This event will be fired, when the cancel button was pressed.
				 */
				cancel: {}
			}
		}
	});

	/**
	 * @override
	 */
	Item.prototype.setShowResetButton = function (bShowResetButton) {
		this.setProperty("showResetButton", bShowResetButton);
		this.changeButtonSettings({
			reset: {visible: bShowResetButton}
		});
		return this;
	};

	/**
	 * @override
	 */
	Item.prototype.setResetButtonEnabled = function (bResetButtonEnabled) {
		this.setProperty("resetButtonEnabled", bResetButtonEnabled);
		this.changeButtonSettings({
			reset: {enabled: bResetButtonEnabled}
		});
		return this;
	};

	/**
	 * @override
	 */
	Item.prototype.setShowConfirmButton = function (bShowConfirmButton) {
		this.setProperty("showConfirmButton", bShowConfirmButton);
		this.changeButtonSettings({
			confirm: {visible: bShowConfirmButton}
		});
		return this;
	};

	/**
	 * @override
	 */
	Item.prototype.setShowCancelButton = function (bShowCancelButton) {
		this.setProperty("showCancelButton", bShowCancelButton);
		this.changeButtonSettings({
			cancel: {visible: bShowCancelButton}
		});
		return this;
	};

	return Item;
});